import React from 'react';
import { Song } from '../types';
import {
  PlayIcon,
  PauseIcon,
  HeartIcon,
  MoreHorizontalIcon,
  SkipBackIcon,
  SkipForwardIcon,
  ShuffleIcon,
  RepeatIcon,
} from './Icons';

interface DesktopPlayerProps {
  song: Song;
  isPlaying: boolean;
  onTogglePlay: () => void;
  isFavorite: boolean;
  onToggleFavorite: () => void;
  currentTime?: number;
  duration?: number;
}

const DesktopPlayer: React.FC<DesktopPlayerProps> = ({
  song,
  isPlaying,
  onTogglePlay,
  isFavorite,
  onToggleFavorite,
  currentTime = 0,
  duration = 0,
}) => {
  const artwork = song.artworkUrl ?? '/images/generated.jpg';
  const subtitle = song.artist ?? 'Suno AI';

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  const progress = duration > 0 ? (currentTime / duration) * 100 : 0;

  return (
    <div className="fixed bottom-0 left-0 right-0 z-50 border-t border-white/10 bg-[#1a1b24]/95 backdrop-blur">
      <div className="mx-auto flex max-w-[1920px] items-center justify-between gap-6 px-6 py-3">
        {/* Left: Song Info */}
        <div className="flex min-w-0 flex-1 items-center gap-4">
          <img
            src={artwork}
            alt={song.title}
            className="h-14 w-14 flex-shrink-0 rounded object-cover"
          />
          <div className="min-w-0 flex-1">
            <p className="truncate text-sm font-semibold text-white">{song.title}</p>
            <p className="truncate text-xs text-gray-400">{subtitle}</p>
          </div>
          <button
            onClick={onToggleFavorite}
            className={`flex-shrink-0 transition-colors ${
              isFavorite ? 'text-pink-500 hover:text-pink-400' : 'text-gray-400 hover:text-white'
            }`}
            aria-label={isFavorite ? 'Убрать из избранного' : 'Добавить в избранное'}
          >
            <HeartIcon className="h-5 w-5" fill={isFavorite ? 'currentColor' : 'none'} />
          </button>
          <button
            className="flex-shrink-0 text-gray-400 transition-colors hover:text-white"
            aria-label="Еще"
          >
            <MoreHorizontalIcon className="h-5 w-5" />
          </button>
        </div>

        {/* Center: Playback Controls */}
        <div className="flex flex-col items-center gap-2">
          <div className="flex items-center gap-4">
            <button
              className="text-gray-400 transition-colors hover:text-white"
              aria-label="Перемешать"
            >
              <ShuffleIcon className="h-5 w-5" />
            </button>
            <button
              className="text-white transition-colors hover:text-gray-300"
              aria-label="Предыдущий трек"
            >
              <SkipBackIcon className="h-6 w-6" />
            </button>
            <button
              onClick={onTogglePlay}
              className="flex h-10 w-10 items-center justify-center rounded-full bg-white text-black transition hover:scale-105"
              aria-label={isPlaying ? 'Пауза' : 'Воспроизвести'}
            >
              {isPlaying ? (
                <PauseIcon className="h-5 w-5" />
              ) : (
                <PlayIcon className="h-5 w-5 ml-0.5" />
              )}
            </button>
            <button
              className="text-white transition-colors hover:text-gray-300"
              aria-label="Следующий трек"
            >
              <SkipForwardIcon className="h-6 w-6" />
            </button>
            <button
              className="text-gray-400 transition-colors hover:text-white"
              aria-label="Повторить"
            >
              <RepeatIcon className="h-5 w-5" />
            </button>
          </div>
          {/* Progress Bar with Waveform */}
          <div className="flex w-[600px] items-center gap-2">
            <span className="text-xs text-gray-400">{formatTime(currentTime)}</span>
            <div className="relative flex-1">
              <div className="h-1 rounded-full bg-gray-700">
                <div
                  className="h-1 rounded-full bg-white transition-all"
                  style={{ width: `${progress}%` }}
                />
              </div>
              {/* Waveform visualization placeholder */}
              <div className="absolute inset-0 flex items-center justify-center gap-0.5 opacity-20">
                {Array.from({ length: 50 }).map((_, i) => (
                  <div
                    key={i}
                    className="w-0.5 bg-white"
                    style={{
                      height: `${Math.random() * 60 + 20}%`,
                    }}
                  />
                ))}
              </div>
            </div>
            <span className="text-xs text-gray-400">{formatTime(duration)}</span>
          </div>
        </div>

        {/* Right: Volume & Queue */}
        <div className="flex items-center gap-3">
          <button
            className="text-gray-400 transition-colors hover:text-white"
            aria-label="Громкость"
          >
            <svg className="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
              <path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z" />
            </svg>
          </button>
          <button
            className="text-gray-400 transition-colors hover:text-white"
            aria-label="Очередь"
          >
            <svg className="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
              <path d="M4 6h16v2H4zm0 5h16v2H4zm0 5h16v2H4z" />
            </svg>
          </button>
        </div>
      </div>
    </div>
  );
};

export default DesktopPlayer;




