import React from 'react';
import { AuthUser } from '../types';
import { LogoIcon, SearchIcon, GenerationsIcon } from './Icons';

interface DesktopHeaderProps {
  user: AuthUser | null;
  onLogin?: () => void;
  onLogout?: () => void;
  onGenerate?: () => void;
  isAuthenticating?: boolean;
}

const DesktopHeader: React.FC<DesktopHeaderProps> = ({
  user,
  onLogin,
  onLogout,
  onGenerate,
  isAuthenticating,
}) => {
  const displayName =
    user?.name ??
    ([user?.telegram_first_name, user?.telegram_last_name].filter(Boolean).join(' ') ||
      (user?.telegram_username ? `@${user.telegram_username}` : 'Гость'));

  return (
    <header className="sticky top-0 z-50 border-b border-white/10 bg-[#1a1b24]/95 backdrop-blur">
      <div className="mx-auto flex max-w-[1920px] items-center justify-between gap-6 px-6 py-4">
        {/* Logo */}
        <div className="flex items-center gap-2">
          <LogoIcon />
          <h1 className="text-2xl font-bold tracking-tighter text-white">Melodist</h1>
        </div>

        {/* Search Bar */}
        <div className="flex-1 max-w-2xl">
          <div className="relative">
            <SearchIcon className="absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2 text-gray-400" />
            <input
              type="text"
              placeholder="Создайте музыку, которую вы представляете..."
              className="w-full rounded-full border border-white/10 bg-white/5 px-12 py-3 text-sm text-white placeholder:text-gray-400 focus:border-purple-500/50 focus:outline-none focus:ring-2 focus:ring-purple-500/20"
            />
          </div>
        </div>

        {/* Right Actions */}
        <div className="flex items-center gap-3">
          {user && onGenerate && (
            <button
              type="button"
              onClick={onGenerate}
              className="flex items-center gap-2 rounded-full border border-white/10 bg-white/5 px-4 py-2 text-sm font-semibold text-white transition hover:bg-white/10"
            >
              <GenerationsIcon className="h-4 w-4" />
              Создать
            </button>
          )}
          {user ? (
            <>
              <div className="text-right">
                <p className="text-sm font-semibold text-white">{displayName}</p>
                {user.is_telegram && user.telegram_username && (
                  <p className="text-xs text-gray-400">@{user.telegram_username}</p>
                )}
              </div>
              {onLogout && (
                <button
                  type="button"
                  onClick={onLogout}
                  className="rounded-full border border-white/10 px-4 py-2 text-sm font-semibold text-gray-300 transition hover:border-purple-400 hover:text-white"
                >
                  Выйти
                </button>
              )}
            </>
          ) : (
            <>
              {onLogin && (
                <button
                  type="button"
                  onClick={onLogin}
                  disabled={isAuthenticating}
                  className="rounded-full border border-white/10 px-4 py-2 text-sm font-semibold text-gray-300 transition hover:border-purple-400 hover:text-white disabled:cursor-not-allowed disabled:opacity-60"
                >
                  {isAuthenticating ? 'Подключаем...' : 'Войти'}
                </button>
              )}
              <button
                type="button"
                onClick={onLogin}
                className="flex items-center gap-2 rounded-full bg-purple-500 px-6 py-2 text-sm font-semibold text-white transition hover:bg-purple-600"
              >
                Начать создавать
                <svg
                  className="h-4 w-4"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M9 5l7 7-7 7"
                  />
                </svg>
              </button>
            </>
          )}
        </div>
      </div>
    </header>
  );
};

export default DesktopHeader;

