import React, {
  useCallback,
  useEffect,
  useMemo,
  useRef,
  useState,
} from 'react';
import BottomNav from './components/BottomNav';
import CreatePage from './components/CreatePage';
import Header from './components/Header';
import HomePage from './components/HomePage';
import LibraryPage from './components/LibraryPage';
import MiniPlayer from './components/MiniPlayer';
import MorePage from './components/MorePage';
import PlayerModal from './components/PlayerModal';
import DesktopHeader from './components/DesktopHeader';
import DesktopSidebar from './components/DesktopSidebar';
import DesktopPlayer from './components/DesktopPlayer';
import DesktopHomePage from './components/DesktopHomePage';
import { AuthUser, Song, GenerationTask } from './types';

declare global {
  interface TelegramWebAppUser {
    id: number;
    username?: string;
    first_name?: string;
    last_name?: string;
    photo_url?: string;
  }

  interface TelegramWebApp {
    initData: string;
    initDataUnsafe: {
      user?: TelegramWebAppUser;
      auth_date?: number;
      query_id?: string;
    };
    ready: () => void;
    expand?: () => void;
  }

  interface TelegramNamespace {
    WebApp?: TelegramWebApp;
  }

  interface Window {
    Telegram?: TelegramNamespace;
    __APP_CANONICAL_ORIGIN__?: string;
    __APP_API_BASE__?: string;
    __APP_API_BASE_PATH__?: string;
    __VKID_APP_ID__?: string | number;
    __VKID_REDIRECT_URL__?: string;
    VKIDSDK?: any;
  }
}

type SunoModel = 'V3_5' | 'V4' | 'V4_5' | 'V4_5PLUS' | 'V5';

interface GenerateMusicRequest {
  prompt: string;
  genre?: string | null;
  mood?: string | null;
  duration?: number;
  customMode?: boolean;
  instrumental?: boolean;
  model?: SunoModel;
  title?: string | null;
  style?: string | null;
}

const DEFAULT_SUNO_MODEL: SunoModel = 'V5';
const POLL_INTERVAL_MS = 5000;
const MAX_POLL_ATTEMPTS = 48;
const FALLBACK_TELEGRAM_BOT_ID = 8426120322;
const FALLBACK_TELEGRAM_LOGIN_BOT = '@slava2voicebot';

type RawSunoSong = Record<string, any>;
type RawSunoResponse = Record<string, any>;

const mapSunoSongToSong = (
  taskId: string,
  index: number,
  raw: RawSunoSong,
): Song => {
  const nowIso = new Date().toISOString();
  const rawId =
    raw?.songId ??
    raw?.song_id ??
    raw?.id ??
    raw?.uuid ??
    `${taskId}-${index + 1}`;
  const title =
    raw?.title ??
    raw?.songName ??
    raw?.name ??
    `Сгенерированный трек ${index + 1}`;
  const artwork =
    raw?.imageUrl ??
    raw?.image_url ??
    raw?.coverImageUrl ??
    raw?.coverUrl ??
    raw?.cover ??
    raw?.image ??
    null;
  const sourceArtwork = raw?.source_image_url ?? raw?.sourceImageUrl ?? null;
  const audioUrl =
    raw?.audioUrl ??
    raw?.audio_url ??
    raw?.downloadUrl ??
    raw?.audioDownloadUrl ??
    raw?.fileUrl ??
    raw?.audio_file_url ??
    null;
  const sourceAudio = raw?.source_audio_url ?? raw?.origin_audio_url ?? null;
  const sourceAudioId =
    raw?.audioId ??
    raw?.source_audio_id ??
    raw?.origin_audio_id ??
    null;
  const streamUrl =
    raw?.streamUrl ??
    raw?.streamingUrl ??
    raw?.stream_audio_url ??
    raw?.source_stream_audio_url ??
    raw?.streaming_audio_url ??
    raw?.audioUrlStreaming ??
    raw?.audioStreamingUrl ??
    null;
  const status =
    raw?.status ??
    raw?.state ??
    raw?.songStatus ??
    raw?.song_status ??
    'pending';
  const duration =
    typeof raw?.duration === 'number'
      ? raw.duration
      : typeof raw?.durationSeconds === 'number'
        ? raw.durationSeconds
        : typeof raw?.audioLengthSeconds === 'number'
          ? raw.audioLengthSeconds
          : null;
  const model =
    raw?.model ??
    raw?.mv ??
    raw?.modelVersion ??
    raw?.version ??
    raw?.songModel ??
    raw?.model_name ??
    null;
  const createdAtRaw =
    raw?.createdAt ?? raw?.created_at ?? raw?.createTime ?? nowIso;

  return {
    id: String(rawId),
    taskId,
    title: String(title),
    artist:
      raw?.artist ??
      raw?.artistName ??
      raw?.author ??
      raw?.creator ??
      'Suno AI',
    prompt: raw?.prompt ?? raw?.lyrics ?? null,
    tags: raw?.tags ?? null,
    artworkUrl: artwork ? String(artwork) : null,
    sourceArtworkUrl: sourceArtwork ? String(sourceArtwork) : null,
    audioUrl: audioUrl ? String(audioUrl) : null,
    sourceAudioUrl: sourceAudio ? String(sourceAudio) : null,
    streamUrl: streamUrl ? String(streamUrl) : null,
    sourceStreamUrl: raw?.source_stream_audio_url
      ? String(raw.source_stream_audio_url)
      : null,
    sourceAudioId: sourceAudioId ? String(sourceAudioId) : null,
    status: String(status),
    model: model ? String(model) : null,
    durationSeconds:
      typeof duration === 'number' ? duration : null,
    createdAt:
      typeof createdAtRaw === 'string'
        ? createdAtRaw
        : nowIso,
  };
};

const normalizeGenerationResponse = (
  payload: RawSunoResponse,
): GenerationTask => {
  const dataObject =
    payload?.data && typeof payload.data === 'object'
      ? payload.data
      : payload;

  console.log('[normalize] Payload:', payload);
  console.log('[normalize] Data object:', dataObject);

  const fallbackTaskId =
    typeof crypto !== 'undefined' && 'randomUUID' in crypto
      ? crypto.randomUUID()
      : `task-${Date.now()}-${Math.random().toString(36).slice(2, 10)}`;

  const taskId =
    dataObject?.taskId ??
    dataObject?.task_id ??
    dataObject?.id ??
    dataObject?.taskUuid ??
    payload?.taskId ??
    payload?.task_id ??
    fallbackTaskId;

  console.log('[normalize] Extracted taskId:', taskId);

  const status =
    dataObject?.callbackType ??
    dataObject?.taskStatus ??
    dataObject?.status ??
    dataObject?.state ??
    payload?.msg ??
    'pending';

  const callbackType =
    typeof dataObject?.callbackType === 'string'
      ? dataObject.callbackType
      : null;

  const responseCode =
    typeof payload?.code === 'number'
      ? payload.code
      : typeof dataObject?.code === 'number'
        ? dataObject.code
        : null;

  const songsRaw =
    Array.isArray(dataObject?.data)
      ? dataObject.data
      : Array.isArray(dataObject?.songs)
        ? dataObject.songs
        : Array.isArray(dataObject?.songInfos)
          ? dataObject.songInfos
          : Array.isArray(dataObject?.songsInfo)
            ? dataObject.songsInfo
            : Array.isArray(dataObject?.tracks)
              ? dataObject.tracks
              : [];

  const normalizedSongs = songsRaw.map((song, index) =>
    mapSunoSongToSong(String(taskId), index, song),
  );

  const nowIso = new Date().toISOString();
  const metaPayload =
    (payload && typeof payload === 'object' && 'meta' in payload
      ? (payload as { meta?: unknown }).meta
      : undefined) ??
    dataObject?.meta ??
    null;

  return {
    id: String(taskId),
    status: String(status),
    callbackType,
    code: responseCode,
    message:
      typeof payload?.msg === 'string'
        ? payload.msg
        : typeof dataObject?.msg === 'string'
          ? dataObject.msg
          : null,
    songs: normalizedSongs,
    createdAt:
      typeof dataObject?.createdAt === 'string'
        ? dataObject.createdAt
        : typeof dataObject?.createTime === 'string'
          ? dataObject.createTime
        : nowIso,
    updatedAt:
      typeof dataObject?.updatedAt === 'string'
        ? dataObject.updatedAt
        : typeof dataObject?.updateTime === 'string'
          ? dataObject.updateTime
        : nowIso,
    meta:
      metaPayload && typeof metaPayload === 'object'
        ? (metaPayload as Record<string, unknown>)
        : null,
  };
};

const mergeSongLists = (current: Song[], incoming: Song[]): Song[] => {
  if (incoming.length === 0) {
    return current;
  }

  const map = new Map<string, Song>();
  current.forEach((song) => {
    map.set(song.id, song);
  });
  incoming.forEach((song) => {
    const previous = map.get(song.id);
    map.set(song.id, previous ? { ...previous, ...song } : song);
  });

  return Array.from(map.values()).sort((a, b) => {
    const aTime = Date.parse(a.createdAt);
    const bTime = Date.parse(b.createdAt);
    return bTime - aTime;
  });
};

const extractGeneratedBy = (task: GenerationTask): string | null => {
  const meta =
    task.meta && typeof task.meta === 'object'
      ? (task.meta as Record<string, unknown>)
      : null;

  if (!meta) {
    return null;
  }

  const candidates = [
    meta.generated_by,
    meta.generatedBy,
    meta.user_name,
    meta.userName,
  ];

  for (const candidate of candidates) {
    if (typeof candidate === 'string') {
      const trimmed = candidate.trim();
      if (trimmed.length > 0) {
        return trimmed;
      }
    }
  }

  return null;
};

const createPlaceholderSong = (task: GenerationTask, generatedBy: string | null): Song => {
  const meta =
    task.meta && typeof task.meta === 'object'
      ? (task.meta as Record<string, unknown>)
      : null;
  const prompt =
    meta && typeof meta.prompt === 'string' && meta.prompt.trim().length > 0
      ? meta.prompt.trim()
      : null;
  const model =
    meta && typeof meta.model === 'string' && meta.model.trim().length > 0
      ? meta.model.trim()
      : null;

  const fallbackTitle = prompt ? prompt : `Генерация трека`;
  const title =
    fallbackTitle.length > 60 ? `${fallbackTitle.slice(0, 57).trim()}…` : fallbackTitle;

  return {
    id: `${task.id}-placeholder`,
    taskId: task.id,
    title,
    artist: generatedBy ?? 'Suno AI',
    prompt,
    tags: null,
    artworkUrl: null,
    sourceArtworkUrl: null,
    audioUrl: null,
    sourceAudioUrl: null,
    streamUrl: null,
    sourceStreamUrl: null,
    sourceAudioId: null,
    status: task.status ?? 'pending',
    model,
    durationSeconds: null,
    createdAt: task.updatedAt ?? task.createdAt ?? new Date().toISOString(),
    isPlaceholder: true,
  };
};

const collectSongsFromTasks = (taskMap: Record<string, GenerationTask>): Song[] => {
  const aggregated: Song[] = [];

  Object.values(taskMap).forEach((task) => {
    const generatedBy = extractGeneratedBy(task);
    const isErrorTask =
      task.status === 'error' ||
      task.callbackType === 'error' ||
      task.message?.toLowerCase().includes('insufficient');

    if (task.songs.length > 0) {
      aggregated.push(
        ...task.songs.map((song) => {
          const artist =
            generatedBy ??
            (typeof song.artist === 'string' && song.artist.trim().length > 0
              ? song.artist
              : 'Suno AI');

          return {
            ...song,
            taskId: song.taskId ?? task.id,
            artist,
            isPlaceholder: false,
          };
        }),
      );
      return;
    }

    if (isErrorTask) {
      return;
    }

    aggregated.push(createPlaceholderSong(task, generatedBy));
  });

  return mergeSongLists([], aggregated);
};

const isSongReady = (song: Song) =>
  Boolean(
    song.streamUrl ??
      song.audioUrl ??
      song.sourceStreamUrl ??
      song.sourceAudioUrl,
  );

const isTaskEffectivelyComplete = (task: GenerationTask) =>
  task.status === 'complete' ||
  task.callbackType === 'complete' ||
  task.status === 'error' ||
  task.callbackType === 'error' ||
  task.songs.every(isSongReady);

interface TelegramLoginWidgetProps {
  botName?: string;
  onAuth: (data: Record<string, unknown>) => void;
  disabled?: boolean;
}

const TelegramLoginWidget: React.FC<TelegramLoginWidgetProps> = ({
  botName,
  onAuth,
  disabled = false,
}) => {
  const loginName = botName?.replace(/^@/, '');

  const containerRef = useRef<HTMLDivElement | null>(null);
  const callbackNameRef = useRef<string | null>(null);

  useEffect(() => {
    const container = containerRef.current;

    if (!container || !loginName) {
      return;
    }

    container.innerHTML = '';

    if (disabled) {
      return;
    }

    const script = document.createElement('script');
    script.src = 'https://telegram.org/js/telegram-widget.js?22';
    script.async = true;
    script.defer = true;

    const callbackName = `telegramAuth_${Math.random().toString(36).slice(2)}`;
    callbackNameRef.current = callbackName;
    (window as unknown as Record<string, unknown>)[callbackName] = (payload: Record<string, unknown>) => {
      onAuth(payload);
    };

    script.setAttribute('data-telegram-login', loginName);
    script.setAttribute('data-size', 'large');
    script.setAttribute('data-radius', '12');
    script.setAttribute('data-userpic', 'false');
    script.setAttribute('data-lang', 'ru');
    script.setAttribute('data-onauth', callbackName);
    script.setAttribute('data-request-access', 'write');

    container.appendChild(script);

    return () => {
      if (callbackNameRef.current) {
        delete (window as unknown as Record<string, unknown>)[callbackNameRef.current];
        callbackNameRef.current = null;
      }

      container.innerHTML = '';
    };
  }, [loginName, onAuth, disabled]);

  return (
    <div
      ref={containerRef}
      className="flex justify-center"
      aria-disabled={disabled}
    />
  );
};

const VKID_SCRIPT_URL = 'https://unpkg.com/@vkid/sdk@2.5.0/dist-sdk/umd/index.js';

interface VKIDAuthTokenPayload {
  access_token: string;
  refresh_token: string;
  token_type: string;
  expires_in: number;
  user_id: string;
  scope?: string | null;
  state?: string | null;
}

interface VKIDAuthPayload {
  device_id: string;
  token: VKIDAuthTokenPayload;
  state?: string | null;
}

interface VKIDLoginButtonProps {
  appId: number;
  redirectUrl: string;
  disabled?: boolean;
  onAuth: (payload: VKIDAuthPayload) => void;
  onError: (message: string) => void;
}

const VKIDLoginButton: React.FC<VKIDLoginButtonProps> = ({
  appId,
  redirectUrl,
  disabled = false,
  onAuth,
  onError,
}) => {
  const containerRef = useRef<HTMLDivElement | null>(null);
  const [isReady, setIsReady] = useState(false);
  const configInitializedRef = useRef(false);
  const widgetRef = useRef<any>(null);

  useEffect(() => {
    // Не загружаем SDK, если VK ID отключен
    if (appId <= 0 || disabled) {
      // Очищаем виджет, если он был создан
      if (widgetRef.current) {
        try {
          widgetRef.current.close();
        } catch (error) {
          // Игнорируем ошибки при закрытии
        }
        widgetRef.current = null;
      }
      if (containerRef.current) {
        containerRef.current.innerHTML = '';
      }
      setIsReady(false);
      return;
    }

    if (typeof window === 'undefined') {
      return;
    }

    if (window.VKIDSDK) {
      setIsReady(true);
      return;
    }

    const existingScript =
      document.querySelector<HTMLScriptElement>('script[data-vkid-sdk]');

    if (existingScript) {
      const handleLoad = () => setIsReady(true);
      const handleError = () => onError('Не удалось загрузить VK ID.');

      existingScript.addEventListener('load', handleLoad);
      existingScript.addEventListener('error', handleError);

      return () => {
        existingScript.removeEventListener('load', handleLoad);
        existingScript.removeEventListener('error', handleError);
      };
    }

    const script = document.createElement('script');
    script.src = VKID_SCRIPT_URL;
    script.async = true;
    script.defer = true;
    script.dataset.vkidSdk = 'true';

    const handleLoad = () => {
      try {
        setIsReady(true);
      } catch (error) {
        console.warn('[VK ID] Load handler error:', error);
      }
    };
    
    const handleError = (error: Event) => {
      // Тихо обрабатываем ошибки загрузки SDK
      console.warn('[VK ID] SDK load error:', error);
      // Не показываем ошибку пользователю, если VK ID не критичен
      // onError('Не удалось загрузить VK ID.');
    };

    script.addEventListener('load', handleLoad);
    script.addEventListener('error', handleError);

    try {
      document.head.appendChild(script);
    } catch (error) {
      console.warn('[VK ID] Failed to append script:', error);
    }

    return () => {
      script.removeEventListener('load', handleLoad);
      script.removeEventListener('error', handleError);
    };
  }, [appId, onError]);

  useEffect(() => {
    if (appId <= 0) {
      return;
    }

    if (typeof window === 'undefined') {
      return;
    }

    if (!isReady || disabled) {
      if (widgetRef.current) {
        try {
          widgetRef.current.close();
        } catch (error) {
          console.warn('VK ID widget close error', error);
        }
        widgetRef.current = null;
      }

      if (containerRef.current) {
        containerRef.current.innerHTML = '';
      }

      return;
    }

    if (!containerRef.current) {
      return;
    }

    const VKID = window.VKIDSDK;

    if (!VKID) {
      onError('VK ID SDK недоступен.');
      return;
    }

    if (!configInitializedRef.current) {
      try {
        VKID.Config.init({
          app: appId,
          redirectUrl,
          responseMode: VKID.ConfigResponseMode.Callback,
          source: VKID.ConfigSource.LOWCODE,
          scope: '',
        });
        configInitializedRef.current = true;
      } catch (error) {
        console.error('VK ID config init error', error);
        onError('Не удалось инициализировать VK ID.');
        return;
      }
    }

    containerRef.current.innerHTML = '';

    let widget: any = null;
    try {
      widget = new VKID.OneTap();
      widgetRef.current = widget;
    } catch (error) {
      console.warn('[VK ID] Failed to create widget:', error);
      return;
    }

    const handleError = (event: unknown) => {
      // Тихо обрабатываем ошибки виджета
      console.warn('[VK ID] Widget error:', event);
      // Не показываем ошибку пользователю
      // let message: string | null = null;
      // if (event && typeof event === 'object') {
      //   const data = event as Record<string, unknown>;
      //   if (typeof data.text === 'string' && data.text.trim().length > 0) {
      //     message = data.text;
      //   } else if (typeof data.error === 'string' && data.error.trim().length > 0) {
      //     message = data.error;
      //   }
      // }
      // onError(message ?? 'Не удалось загрузить VK ID.');
    };

    const handleLogin = async (payload: unknown) => {
      if (!payload || typeof payload !== 'object') {
        onError('Ответ VK ID не содержит данных авторизации.');
        return;
      }

      const data = payload as Record<string, unknown>;
      const code = typeof data.code === 'string' ? data.code : null;
      const deviceId =
        typeof data.device_id === 'string' ? data.device_id : data.device_id != null ? String(data.device_id) : null;

      if (!code || !deviceId) {
        onError('VK ID вернул неполные данные авторизации.');
        return;
      }

      try {
        const result = await VKID.Auth.exchangeCode(code, deviceId);

        const scope =
          typeof result?.scope === 'string'
            ? result.scope
            : typeof data.scope === 'string'
              ? data.scope
              : undefined;

        const state =
          typeof result?.state === 'string'
            ? result.state
            : typeof data.state === 'string'
              ? data.state
              : null;

        const userId =
          result?.user_id != null
            ? String(result.user_id)
            : data.user_id != null
              ? String(data.user_id)
              : '';

        const accessToken = typeof result?.access_token === 'string' ? result.access_token : '';
        const refreshToken = typeof result?.refresh_token === 'string' ? result.refresh_token : '';
        const tokenType = typeof result?.token_type === 'string' ? result.token_type : 'Bearer';
        const expiresInRaw = result?.expires_in;
        const expiresIn = Number.isFinite(expiresInRaw) ? Number(expiresInRaw) : 0;

        if (!accessToken || !refreshToken || userId === '') {
          onError('Не удалось получить токены VK ID.');
          return;
        }

        onAuth({
          device_id: deviceId,
          state,
          token: {
            access_token: accessToken,
            refresh_token: refreshToken,
            token_type: tokenType,
            expires_in: expiresIn,
            user_id: userId,
            scope,
            state,
          },
        });
      } catch (error) {
        console.error('VK ID exchange error', error);
        onError('Не удалось завершить вход через VK ID.');
      }
    };

    widget
      .render({
        container: containerRef.current,
        showAlternativeLogin: true,
      })
      .on(VKID.WidgetEvents.ERROR, handleError)
      .on(VKID.OneTapInternalEvents.LOGIN_SUCCESS, handleLogin);

    return () => {
      widget.off(VKID.WidgetEvents.ERROR, handleError);
      widget.off(VKID.OneTapInternalEvents.LOGIN_SUCCESS, handleLogin);
      widget.close();
      widgetRef.current = null;
    };
  }, [appId, redirectUrl, isReady, disabled, onAuth, onError]);

  if (appId <= 0) {
    return null;
  }

  return <div ref={containerRef} className="flex justify-center" />;
};

interface BrowserTelegramLoginProps {
  botName?: string;
  botId?: number;
  onAuth: (data: Record<string, unknown>) => void;
  onVKIDAuth: (payload: VKIDAuthPayload) => void;
  onVKIDError: (message: string) => void;
  isSubmitting: boolean;
  authError: string | null;
  onOpenTelegramApp: () => void;
  vkidAppId: number;
  vkidRedirectUrl: string;
  showDemoLogin?: boolean;
  onDemoLogin?: () => void;
}

const BrowserTelegramLogin: React.FC<BrowserTelegramLoginProps> = ({
  botName,
  botId,
  onAuth,
  onVKIDAuth,
  onVKIDError,
  isSubmitting,
  authError,
  onOpenTelegramApp,
  vkidAppId,
  vkidRedirectUrl,
  showDemoLogin = false,
  onDemoLogin,
}) => {
  const resolvedBotName =
    (botName && botName.trim().length > 0 ? botName.trim() : null) ??
    FALLBACK_TELEGRAM_LOGIN_BOT;
  const resolvedBotId =
    typeof botId === 'number' && !Number.isNaN(botId) ? botId : FALLBACK_TELEGRAM_BOT_ID;
  const normalizedBotName = resolvedBotName.replace(/^@/, '');

  const openWindowOrRedirect = useCallback((url: string) => {
    if (typeof window === 'undefined') {
      return null;
    }

    const newWindow = window.open(url, '_blank', 'noopener,noreferrer');

    if (!newWindow) {
      window.location.href = url;
    }

    return newWindow;
  }, []);

  const handleOpen = useCallback(() => {
    if (typeof window === 'undefined') {
      onOpenTelegramApp();
      return;
    }

    if (resolvedBotId) {
      const origin = encodeURIComponent(window.location.origin);
      const url = `https://oauth.telegram.org/auth?bot_id=${resolvedBotId}&origin=${origin}&embed=1&request_access=write`;
      openWindowOrRedirect(url);
      return;
    }

    if (normalizedBotName) {
      const deepLink = `tg://resolve?domain=${normalizedBotName}&start=webapp`;
      const httpsLink = `https://t.me/${normalizedBotName}?start=webapp`;
      const opened = openWindowOrRedirect(deepLink);

      window.setTimeout(() => {
        if (!opened || opened.closed) {
          openWindowOrRedirect(httpsLink);
        }
      }, 400);
      return;
    }

    onOpenTelegramApp();
  }, [botId, normalizedBotName, onOpenTelegramApp, openWindowOrRedirect]);

  return (
    <div className="space-y-6">
      {authError && (
        <div className="rounded-lg border border-red-500/40 bg-red-500/10 px-3 py-2 text-sm text-red-200">
          {authError}
        </div>
      )}

      <div className="space-y-3">
        <TelegramLoginWidget
          botName={resolvedBotName}
          onAuth={onAuth}
          disabled={isSubmitting}
        />
        <p className="text-center text-xs text-gray-400">
          Telegram откроет окно подтверждения. После разрешения входа вы попадёте в приложение.
        </p>
        {isSubmitting && (
          <p className="text-center text-xs text-gray-400">
            Проверяем данные Telegram...
          </p>
        )}
      </div>

      {vkidAppId > 0 && (
        <div className="space-y-3">
          <div className="flex items-center justify-center">
            <span className="text-xs uppercase tracking-widest text-gray-500">или</span>
          </div>
          <VKIDLoginButton
            appId={vkidAppId}
            redirectUrl={vkidRedirectUrl}
            disabled={isSubmitting}
            onAuth={onVKIDAuth}
            onError={onVKIDError}
          />
          <p className="text-center text-xs text-gray-400">
            Вход через VK ID. После авторизации мы сохраним только ваше имя и идентификатор.
          </p>
        </div>
      )}

      <div className="flex justify-center">
        <button
          type="button"
          onClick={handleOpen}
          className="text-xs font-semibold text-purple-300 underline underline-offset-4 transition hover:text-purple-200"
        >
          Не открывается? Открыть Telegram вручную
        </button>
      </div>

      {showDemoLogin && (
        <div className="space-y-3">
          <div className="flex justify-center">
            <button
              type="button"
              onClick={() => {
                if (onDemoLogin) {
                  void onDemoLogin();
                }
              }}
              disabled={isSubmitting}
              className="w-full rounded-full border border-emerald-400/60 px-4 py-3 text-sm font-semibold text-emerald-200 transition hover:border-emerald-400 hover:text-white disabled:cursor-not-allowed disabled:opacity-60"
            >
              Попробовать демо-режим
            </button>
          </div>
          <p className="text-center text-xs text-gray-500">
            Мы создадим тестовый аккаунт, чтобы вы могли посмотреть интерфейс без Telegram.
          </p>
        </div>
      )}

      <p className="text-center text-xs text-gray-500">
        Продолжая, вы соглашаетесь с нашими{' '}
        <a href="#" className="text-purple-300 hover:underline">
          Условиями обслуживания
        </a>{' '}
        и{' '}
        <a href="#" className="text-purple-300 hover:underline">
          Политикой конфиденциальности
        </a>
        .
      </p>
    </div>
  );
};

type AuthStatus = 'loading' | 'authenticated' | 'unauthenticated';

const App: React.FC = () => {
  const [activeTab, setActiveTab] = useState('home');
  const [songs, setSongs] = useState<Song[]>([]);
  const [publicSongs, setPublicSongs] = useState<Song[]>([]);
  const [tasks, setTasks] = useState<Record<string, GenerationTask>>({});
  const [activeTaskId, setActiveTaskId] = useState<string | null>(null);
  const [currentSong, setCurrentSong] = useState<Song | null>(null);
  const [isPlaying, setIsPlaying] = useState(false);
  const [favorites, setFavorites] = useState<Song[]>([]);
  const audioRef = useRef<HTMLAudioElement | null>(null);
  const pollTimers = useRef<Map<string, number>>(new Map());
  const [playbackError, setPlaybackError] = useState<string | null>(null);
  const [isPlayerModalOpen, setPlayerModalOpen] = useState(false);
  const [isGeneratingMusic, setIsGeneratingMusic] = useState(false);
  const [generationError, setGenerationError] = useState<string | null>(null);
  const [extendingSongId, setExtendingSongId] = useState<string | null>(null);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);

  const [authStatus, setAuthStatus] = useState<AuthStatus>('loading');
  const [user, setUser] = useState<AuthUser | null>(null);
  const [token, setToken] = useState<string | null>(null);
  const [authError, setAuthError] = useState<string | null>(null);
  const [isTelegramContext, setIsTelegramContext] = useState(false);
  const [isAuthDialogOpen, setIsAuthDialogOpen] = useState(false);
  const [isDesktop, setIsDesktop] = useState(false);
  const isAuthenticated = authStatus === 'authenticated' && !!user;
  const env =
    ((import.meta as unknown as { env?: Record<string, string | undefined> }).env) ??
    {};

  const isDevBuild = Boolean((import.meta as unknown as { env?: Record<string, unknown> }).env?.DEV);
  const demoFlagFromEnv =
    (env.VITE_ENABLE_DEMO_LOGIN ?? '').trim().toLowerCase() === 'true';
  const [demoLoginAvailable, setDemoLoginAvailable] = useState<boolean>(
    isDevBuild || demoFlagFromEnv,
  );

  useEffect(() => {
    if (typeof window !== 'undefined' && window.location.hostname !== 'sunoai.ru') {
      setDemoLoginAvailable(true);
    }
  }, []);

  // Detect desktop viewport
  useEffect(() => {
    const checkDesktop = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };
    checkDesktop();
    window.addEventListener('resize', checkDesktop);
    return () => window.removeEventListener('resize', checkDesktop);
  }, []);

  const telegramLoginBotRaw = (env.VITE_TELEGRAM_LOGIN_BOT ?? '').trim();
  const telegramLoginBot =
  telegramLoginBotRaw.length > 0 ? telegramLoginBotRaw : FALLBACK_TELEGRAM_LOGIN_BOT;

  const envBotIdRaw = env.VITE_TELEGRAM_BOT_ID ?? '';
  const envBotIdParsed = envBotIdRaw !== '' ? Number(envBotIdRaw) : undefined;
  const initialBotId = !Number.isNaN(envBotIdParsed ?? NaN)
    ? envBotIdParsed
    : FALLBACK_TELEGRAM_BOT_ID;

  const [loginBotName, setLoginBotName] = useState<string | undefined>(telegramLoginBot);
  const [loginBotId, setLoginBotId] = useState<number | undefined>(initialBotId);

  const vkidAppId = useMemo(() => {
    // VK ID отключен по умолчанию, можно включить через VITE_VKID_APP_ID
    const candidates: Array<string | number | undefined | null> = [
      env.VITE_VKID_APP_ID,
      typeof window !== 'undefined' ? window.__VKID_APP_ID__ : undefined,
      // 54308149, // Отключено по умолчанию
    ];

    for (const candidate of candidates) {
      if (candidate === undefined || candidate === null) {
        continue;
      }

      const parsed = Number(candidate);

      if (!Number.isNaN(parsed) && parsed > 0) {
        return parsed;
      }
    }

    return 0;
  }, [env.VITE_VKID_APP_ID]);

  const vkidRedirectUrl = useMemo(() => {
    const fallback = (() => {
      if (typeof window === 'undefined') {
        return 'https://sunoai.ru';
      }

      const redirectCandidates = [
        window.__VKID_REDIRECT_URL__,
        window.__APP_CANONICAL_ORIGIN__,
        window.location?.origin,
      ];

      for (const value of redirectCandidates) {
        if (typeof value === 'string' && value.trim().length > 0) {
          return value.trim();
        }
      }

      return 'https://sunoai.ru';
    })();

    const candidate = env.VITE_VKID_REDIRECT_URL;

    if (typeof candidate === 'string' && candidate.trim().length > 0) {
      return candidate.trim();
    }

    return fallback;
  }, [env.VITE_VKID_REDIRECT_URL]);

  const favoritesStorageKey = useMemo(() => {
    const userId = user?.id ? `user-${user.id}` : 'guest';
    return `slava:favorites:${userId}`;
  }, [user?.id]);

  const rawApiBaseUrl = (env.VITE_API_BASE_URL ?? '').trim();

  const apiBaseCandidates = useMemo(() => {
    const orderedBases: string[] = [];
    let explicitBase: string | null = null;
    const addBase = (value: string, priority = false) => {
      const normalized = value.replace(/\/+$/, '');
      if (!orderedBases.includes(normalized)) {
        if (priority) {
          orderedBases.unshift(normalized);
        } else {
          orderedBases.push(normalized);
        }
      }
    };

    if (rawApiBaseUrl.length > 0) {
      const normalized = rawApiBaseUrl.replace(/\/+$/, '');
      const base = normalized.endsWith('/api') ? normalized : `${normalized}/api`;
      addBase(base);
      addBase(base.replace(/\/api$/, '/backend/public/index.php/api'));
      addBase(base.replace(/\/api$/, '/backend/public/api'));
    }

    if (typeof window !== 'undefined') {
      const explicitBasePath = (window.__APP_API_BASE_PATH__ ?? '').trim();
      if (explicitBasePath.length > 0) {
        explicitBase = explicitBasePath.replace(/\/+$/, '');
        addBase(explicitBase, true);
      }

      const addOriginBases = (origin: string) => {
        const normalizedOrigin = origin.replace(/\/+$/, '');
        addBase(`${normalizedOrigin}/backend/public/index.php/api`);
        addBase(`${normalizedOrigin}/backend/public/api`);
        addBase(`${normalizedOrigin}/api`);
      };

      // Если мы в dev режиме (localhost:5173), используем production URL для API
      const isDevMode = window.location.hostname === 'localhost' && (window.location.port === '5173' || window.location.port === '');
      const productionOrigin = 'https://sunoai.ru';
      
      if (isDevMode) {
        console.log('[api] Dev mode detected, using production API URL');
        addOriginBases(productionOrigin);
      } else {
        addOriginBases(window.location.origin);
      }

      const canonicalOrigin =
        (window.__APP_API_BASE__ ?? window.__APP_CANONICAL_ORIGIN__ ?? '').trim();
      if (canonicalOrigin.length > 0) {
        addOriginBases(canonicalOrigin);
      }

      try {
        const currentScript = document.querySelector(
          'script[type="module"][src*="assets/index.js"]',
        ) as HTMLScriptElement | null;
        if (currentScript?.src) {
          addOriginBases(new URL(currentScript.src).origin);
        }
      } catch {
        // ignore URL parsing issues
      }
    }

    if (explicitBase) {
      return [explicitBase];
    }

    return orderedBases;
  }, [rawApiBaseUrl]);

  const apiFetch = useCallback(
    async (path: string, init: RequestInit = {}, overrideToken?: string | null) => {
      let lastError: Error | null = null;
      const fullUrl = `${apiBaseCandidates[0] || ''}${path.startsWith('/') ? path : `/${path}`}`;
      console.log(`[api] Making request to: ${fullUrl}`);

      for (const base of apiBaseCandidates) {
        const url = `${base}${path.startsWith('/') ? path : `/${path}`}`;
        console.log(`[api] Trying base: ${base}, full URL: ${url}`);
        
        const headers = new Headers(init.headers ?? {});
        headers.set('Accept', 'application/json');

        const authToken = overrideToken === undefined ? token : overrideToken;

        if (authToken) {
          headers.set('Authorization', `Bearer ${authToken}`);
          headers.set('X-Auth-Token', authToken);
        }

        if (init.body && !(init.body instanceof FormData) && !headers.has('Content-Type')) {
          headers.set('Content-Type', 'application/json');
        }

        try {
          const response = await fetch(
            url,
            {
              ...init,
              headers,
            },
          );

          const raw = await response.text();
          let data: any = null;

          if (raw) {
            try {
              data = JSON.parse(raw);
            } catch {
              data = raw;
            }
          }

          console.log(`[api] Response status: ${response.status}, URL: ${url}`);

          if (!response.ok) {
            const status = response.status;
            const message =
              typeof data === 'object' && data !== null && 'message' in data
                ? String((data as { message: unknown }).message)
                : response.statusText;

            lastError = new Error(message || 'Не удалось выполнить запрос.');
            Object.assign(lastError, { status });

            // Для 404 пробуем следующий базовый URL, но если это последний - выбрасываем ошибку
            if (status === 404) {
              const isLastBase = apiBaseCandidates.indexOf(base) === apiBaseCandidates.length - 1;
              if (isLastBase) {
                console.error(`[api] All API bases returned 404. Last error:`, lastError);
                throw lastError;
              }
              console.log(`[api] 404 on ${url}, trying next base...`);
              continue;
            }

            // Для других 4xx ошибок выбрасываем сразу
            if (status >= 400 && status < 500) {
              console.error(`[api] Client error ${status} on ${url}:`, lastError);
              throw lastError;
            }

            // Для 5xx пробуем следующий базовый URL
            continue;
          }

          console.log(`[api] Success on ${url}`);
          return data ?? {};
        } catch (error) {
          // Если это не 404 и не последний базовый URL, пробуем следующий
          const isLastBase = apiBaseCandidates.indexOf(base) === apiBaseCandidates.length - 1;
          if (error instanceof Error && (error as any).status === 404 && !isLastBase) {
            console.log(`[api] Error on ${url}, trying next base...`);
            lastError = error;
            continue;
          }
          
          lastError = error instanceof Error ? error : new Error(String(error));
          
          // Если это последний базовый URL, выбрасываем ошибку
          if (isLastBase) {
            console.error(`[api] All API bases failed. Last error:`, lastError);
            throw lastError;
          }
          
          continue;
        }
      }

      throw (
        lastError ??
        new Error('Не удалось выполнить запрос: все доступные адреса API недоступны.')
      );
    },
    [apiBaseCandidates, token],
  );

  useEffect(() => {
    let cancelled = false;

    if (loginBotName && (loginBotId || telegramLoginBot?.length)) {
      return () => {
        cancelled = true;
      };
    }

    const loadAuthConfig = async () => {
      try {
        const data = await apiFetch('/config/auth');

        if (cancelled || !data || typeof data !== 'object') {
          return;
        }

        const remoteBot = (data as Record<string, unknown>).telegram_login_bot ?? null;
        const remoteBotId = (data as Record<string, unknown>).telegram_bot_id ?? null;
        const remoteDemoEnabled = (data as Record<string, unknown>).demo_enabled ?? null;

        if (typeof remoteBot === 'string' && remoteBot.trim().length > 0) {
          setLoginBotName(remoteBot.startsWith('@') ? remoteBot : `@${remoteBot}`);
        } else {
          setLoginBotName((current) =>
            current ?? FALLBACK_TELEGRAM_LOGIN_BOT,
          );
        }

        if (remoteBotId !== null && !Number.isNaN(Number(remoteBotId))) {
          setLoginBotId(Number(remoteBotId));
        } else {
          setLoginBotId((current) =>
            current !== undefined ? current : FALLBACK_TELEGRAM_BOT_ID,
          );
        }

        if (typeof remoteDemoEnabled === 'boolean') {
          setDemoLoginAvailable(remoteDemoEnabled);
        }
      } catch {
        // ignore
      }
    };

    void loadAuthConfig();

    return () => {
      cancelled = true;
    };
  }, [apiFetch]);

  useEffect(() => {
    const audio = new Audio();
    audioRef.current = audio;

    const handleEnded = () => {
      setIsPlaying(false);
    };

    const handleError = () => {
      setPlaybackError('Не удалось воспроизвести трек. Попробуйте позже.');
    };

    audio.addEventListener('ended', handleEnded);
    audio.addEventListener('error', handleError);

    return () => {
      audio.removeEventListener('ended', handleEnded);
      audio.removeEventListener('error', handleError);
      audio.pause();
      audioRef.current = null;
    };
  }, []);

  useEffect(() => {
    return () => {
      pollTimers.current.forEach((timeoutId) => window.clearTimeout(timeoutId));
      pollTimers.current.clear();
    };
  }, []);

  useEffect(() => {
    const allSongs = collectSongsFromTasks(tasks);
    
    // Фильтруем треки по текущему пользователю
    // Треки из задач должны иметь user_id в метаданных задачи
    // Если пользователь не авторизован, не показываем треки
    if (!user) {
      setSongs([]);
      return;
    }
    
    const userSongs = allSongs.filter((song) => {
      // Если трек из задачи, проверяем метаданные задачи
      if (song.taskId) {
        const task = tasks[song.taskId];
        // Проверяем user_id в метаданных - это основной способ идентификации
        if (task?.meta?.user_id) {
          return task.meta.user_id === user.id;
        }
        // Если нет user_id в метаданных, трек не принадлежит пользователю
        // (старые задачи могут не иметь user_id, их не показываем)
        return false;
      }
      // Если трек без taskId, он может быть из БД
      // Треки из БД уже отфильтрованы на сервере по user_id
      // Но на всякий случай оставляем их (они будут отфильтрованы при загрузке из БД)
      return true;
    });
    
    setSongs(userSongs);
  }, [tasks, user]);

  // Загружаем треки из БД при авторизации
  useEffect(() => {
    if (!isAuthenticated || !token) {
      return;
    }

    let cancelled = false;

    const loadSongsFromDatabase = async () => {
      try {
        console.log('[songs] Loading songs from database...');
        const response = await apiFetch('/songs', {
          method: 'GET',
        }, token);

        console.log('[songs] Songs loaded:', response);
        
        if (cancelled) return;

        const dbSongs = Array.isArray(response?.songs) ? response.songs : [];
        
        // Преобразуем треки из БД в формат Song
        const normalizedSongs: Song[] = dbSongs.map((dbSong: any) => {
          // Используем suno_song_id как основной ID для избежания дублирования
          const songId = dbSong.suno_song_id 
            ? String(dbSong.suno_song_id)
            : (dbSong.id ? String(dbSong.id) : `db-${dbSong.id ?? Date.now()}-${Math.random()}`);
          
          return {
            id: songId,
            taskId: dbSong.task_id ?? `task-${dbSong.id ?? songId}`,
            title: dbSong.title ?? 'Untitled',
            artist: dbSong.artist ?? null,
            prompt: dbSong.prompt ?? null,
            tags: dbSong.tags ?? null,
            artworkUrl: dbSong.artwork_url ?? dbSong.source_artwork_url ?? null,
            audioUrl: dbSong.audio_url ?? dbSong.source_audio_url ?? null,
            streamUrl: dbSong.stream_url ?? dbSong.source_stream_url ?? dbSong.audio_url ?? null,
            sourceAudioUrl: dbSong.source_audio_url ?? dbSong.audio_url ?? null,
            sourceStreamUrl: dbSong.source_stream_url ?? dbSong.stream_url ?? dbSong.audio_url ?? null,
            sourceArtworkUrl: dbSong.source_artwork_url ?? dbSong.artwork_url ?? null,
            durationSeconds: dbSong.duration_seconds ?? null,
            model: dbSong.model ?? 'V5',
            createdAt: dbSong.created_at ?? new Date().toISOString(),
            isPublic: dbSong.is_public ?? false,
            publishedAt: dbSong.published_at ?? null,
            playsCount: dbSong.plays_count ?? 0,
            likesCount: dbSong.likes_count ?? 0,
          };
        });

        // Объединяем треки из БД с треками из задач
        setSongs((currentSongs) => {
          // Создаем Map для быстрого поиска по уникальному ключу
          const songsMap = new Map<string, Song>();
          
          // Сначала добавляем треки из задач
          // Они уже отфильтрованы по пользователю в useEffect выше
          currentSongs.forEach((song) => {
            const key = song.id || song.audioUrl || `${song.taskId}-${song.title}`;
            if (key) {
              songsMap.set(key, song);
            }
          });
          
          // Затем добавляем/обновляем треки из БД (они уже отфильтрованы по user_id на сервере)
          normalizedSongs.forEach((dbSong) => {
            // Пробуем найти по разным ключам
            const keys = [
              dbSong.id,
              dbSong.audioUrl,
              dbSong.streamUrl,
              `${dbSong.taskId}-${dbSong.title}`,
            ].filter(Boolean) as string[];
            
            let found = false;
            for (const key of keys) {
              if (songsMap.has(key)) {
                // Обновляем существующий трек данными из БД
                songsMap.set(key, { ...songsMap.get(key)!, ...dbSong });
                found = true;
                break;
              }
            }
            
            if (!found && keys.length > 0) {
              // Добавляем новый трек из БД
              songsMap.set(keys[0], dbSong);
            }
          });

          return Array.from(songsMap.values());
        });
      } catch (error) {
        console.error('[songs] Error loading songs from database:', error);
      }
    };

    void loadSongsFromDatabase();

    return () => {
      cancelled = true;
    };
  }, [isAuthenticated, token, apiFetch]);

  useEffect(() => {
    const audio = audioRef.current;
    if (!audio) {
      return;
    }

    const source =
      currentSong?.streamUrl ??
      currentSong?.audioUrl ??
      currentSong?.sourceStreamUrl ??
      currentSong?.sourceAudioUrl ??
      '';

    if (!source) {
      audio.pause();
      if (audio.src) {
        audio.removeAttribute('src');
      }
      return;
    }

    if (audio.src !== source) {
      audio.src = source;
    }

    if (isPlaying) {
      audio
        .play()
        .then(() => {
          setPlaybackError(null);
        })
        .catch((error) => {
          console.error('Audio playback error', error);
          setIsPlaying(false);
          setPlaybackError('Не удалось начать воспроизведение. Попробуйте позже.');
        });
    } else {
      audio.pause();
    }
  }, [currentSong, isPlaying]);

  useEffect(() => {
    if (!currentSong && songs.length > 0) {
      setCurrentSong(songs[0]);
      return;
    }

    if (currentSong) {
      const updated = songs.find((song) => song.id === currentSong.id);
      if (updated && updated !== currentSong) {
        setCurrentSong(updated);
      }
    }
  }, [songs, currentSong]);

  useEffect(() => {
    if (typeof window === 'undefined') {
      return;
    }

    try {
      const raw = localStorage.getItem(favoritesStorageKey);
      if (raw) {
        const parsed: Song[] = JSON.parse(raw);
        setFavorites(parsed);
      } else {
        setFavorites([]);
      }
    } catch (error) {
      console.warn('Не удалось загрузить избранные треки', error);
      setFavorites([]);
    }
  }, [favoritesStorageKey]);

  useEffect(() => {
    if (typeof window === 'undefined') {
      return;
    }

    try {
      localStorage.setItem(favoritesStorageKey, JSON.stringify(favorites));
    } catch (error) {
      console.warn('Не удалось сохранить избранные треки', error);
    }
  }, [favorites, favoritesStorageKey]);

  useEffect(() => {
    setFavorites((previous) =>
      previous.map((favorite) => {
        const latest = songs.find((song) => song.id === favorite.id);
        return latest ? { ...favorite, ...latest } : favorite;
      }),
    );
  }, [songs]);

  // ESC key handler for create modal
  useEffect(() => {
    if (!isCreateModalOpen) return;

    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === 'Escape' && !isGeneratingMusic) {
        setIsCreateModalOpen(false);
      }
    };

    document.addEventListener('keydown', handleEscape);
    return () => document.removeEventListener('keydown', handleEscape);
  }, [isCreateModalOpen, isGeneratingMusic]);

  const applyTaskUpdate = useCallback((task: GenerationTask) => {
    const timestamp =
      task.updatedAt ?? new Date().toISOString();

    setTasks((previous) => ({
      ...previous,
      [task.id]: { ...task, updatedAt: timestamp },
    }));

    setCurrentSong((existing) => {
      if (!existing) {
        return task.songs[0] ?? null;
      }

      const match = task.songs.find((song) => song.id === existing.id);

      if (match) {
        return { ...existing, ...match };
      }

      return existing;
    });
  }, []);

  const refreshTask = useCallback(
    async (taskId: string, attempt = 0) => {
      try {
        const response = await apiFetch(
          `/music/tasks/${encodeURIComponent(taskId)}?refresh=1`,
        );
        const task = normalizeGenerationResponse(response);
        applyTaskUpdate(task);

        if (!isTaskEffectivelyComplete(task) && attempt < MAX_POLL_ATTEMPTS) {
          const timeoutId = window.setTimeout(() => {
            void refreshTask(taskId, attempt + 1);
          }, POLL_INTERVAL_MS);
          pollTimers.current.set(taskId, timeoutId);
        } else {
          const existingTimeout = pollTimers.current.get(taskId);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(taskId);
          }
        }
      } catch (error) {
        console.error('Не удалось обновить статус генерации', error);
        if (attempt < MAX_POLL_ATTEMPTS) {
          const timeoutId = window.setTimeout(() => {
            void refreshTask(taskId, attempt + 1);
          }, POLL_INTERVAL_MS * 2);
          pollTimers.current.set(taskId, timeoutId);
        }
      }
    },
    [apiFetch, applyTaskUpdate],
  );

  const handleExtendSong = useCallback(
    async (song: Song) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация.');
      }

      const audioId = song.sourceAudioId ?? song.id;
      const model = song.model ?? DEFAULT_SUNO_MODEL;

      if (!audioId) {
        setGenerationError('Не удалось определить идентификатор трека для продления.');
        return;
      }

      setExtendingSongId(song.id);
      setGenerationError(null);

      try {
        const response = await apiFetch('/music/extend', {
          method: 'POST',
          body: JSON.stringify({
            defaultParamFlag: false,
            audioId,
            model,
          }),
        });

        const task = normalizeGenerationResponse(response);
        applyTaskUpdate(task);
        setActiveTaskId(task.id);

        if (!isTaskEffectivelyComplete(task)) {
          const existingTimeout = pollTimers.current.get(task.id);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(task.id);
          }

          const timeoutId = window.setTimeout(() => {
            void refreshTask(task.id, 1);
          }, POLL_INTERVAL_MS);

          pollTimers.current.set(task.id, timeoutId);
        }
      } catch (error) {
        console.error('Ошибка продления трека', error);
        const message =
          error instanceof Error
            ? error.message
            : 'Не удалось продлить трек.';
        setGenerationError(message);
      } finally {
        setExtendingSongId(null);
      }
    },
    [apiFetch, applyTaskUpdate, isAuthenticated, refreshTask],
  );

  const fetchProfile = useCallback(
    async (authToken: string, telegram = false) => {
      const data = await apiFetch('/auth/me', {}, authToken);

      if (!data || typeof data !== 'object' || !('user' in data)) {
        throw new Error('Не удалось загрузить данные пользователя.');
      }

      const userData = (data as { user: AuthUser }).user;
      setUser(userData);
      setToken(authToken);
      localStorage.setItem('authToken', authToken);
      setIsTelegramContext(Boolean(userData?.is_telegram) || telegram);
      setAuthStatus('authenticated');
      setAuthError(null);
      setIsAuthDialogOpen(false);
    },
    [apiFetch],
  );

  const handleGenerateMusic = useCallback(
    async (options: GenerateMusicRequest) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация.');
      }

      setIsGeneratingMusic(true);
      setGenerationError(null);

      const payload = {
        customMode: options.customMode ?? false,
        instrumental: options.instrumental ?? false,
        model: options.model ?? DEFAULT_SUNO_MODEL,
        prompt: options.prompt,
        genre: options.genre ?? null,
        mood: options.mood ?? null,
        duration: options.duration ?? null,
        title: options.title ?? null,
        style: options.style ?? null,
      };

      try {
        console.log('[generate] Sending request with payload:', payload);
        const response = await apiFetch('/music/generate', {
          method: 'POST',
          body: JSON.stringify(payload),
        });

        console.log('[generate] Response received:', response);
        const task = normalizeGenerationResponse(response);
        console.log('[generate] Normalized task:', task);
        
        if (!task.id || task.id === 'undefined') {
          console.error('[generate] Invalid task ID:', task);
          throw new Error('Сервер не вернул идентификатор задачи. Попробуйте снова.');
        }
        
        applyTaskUpdate(task);
        setActiveTaskId(task.id);

        if (token) {
          try {
            await fetchProfile(token, true);
          } catch (profileError) {
            console.warn('Не удалось обновить профиль после генерации', profileError);
          }
        }

        if (!isTaskEffectivelyComplete(task)) {
          const existingTimeout = pollTimers.current.get(task.id);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(task.id);
          }

          const timeoutId = window.setTimeout(() => {
            void refreshTask(task.id, 1);
          }, POLL_INTERVAL_MS);

          pollTimers.current.set(task.id, timeoutId);
        }
      } catch (error) {
        console.error('Ошибка запуска генерации музыки', error);
        if (error && typeof error === 'object' && 'status' in error && (error as { status?: number }).status === 401) {
          localStorage.removeItem('authToken');
          setToken(null);
          setUser(null);
          setAuthStatus('unauthenticated');
          setAuthError('Сессия истекла. Войдите снова, чтобы продолжить.');
          setIsAuthDialogOpen(true);
        }
        
        let message = 'Не удалось запустить генерацию трека.';
        if (error instanceof Error) {
          message = error.message;
          
          // Обрабатываем специфичные ошибки от Suno API
          if (message.includes('insufficient') || message.includes('credits') || (error as any)?.status === 429) {
            message = 'Недостаточно кредитов на аккаунте Suno API. Пополните баланс.';
          } else if (message.includes('429')) {
            message = 'Превышен лимит запросов. Попробуйте позже.';
          } else if (message.includes('taskId')) {
            message = 'Сервер не смог получить идентификатор задачи. Попробуйте снова.';
          }
        }
        
        setGenerationError(message);
        throw error;
      } finally {
        setIsGeneratingMusic(false);
      }
    },
    [apiFetch, applyTaskUpdate, fetchProfile, isAuthenticated, refreshTask, token],
  );

  const handleCoverGeneration = useCallback(
    async (audioFile: File, options: { prompt?: string; title?: string; tags?: string; model?: string }) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация.');
      }

      setIsGeneratingMusic(true);
      setGenerationError(null);

      const formData = new FormData();
      formData.append('audio_file', audioFile);
      if (options.prompt) formData.append('prompt', options.prompt);
      if (options.title) formData.append('title', options.title);
      if (options.tags) formData.append('tags', options.tags);
      if (options.model) formData.append('model', options.model);

      try {
        const response = await apiFetch('/music/cover', {
          method: 'POST',
          body: formData,
          headers: {}, // Let browser set Content-Type with boundary for FormData
        }, token);

        const task = normalizeGenerationResponse(response);
        
        if (!task.id || task.id === 'undefined') {
          throw new Error('Сервер не вернул идентификатор задачи. Попробуйте снова.');
        }
        
        applyTaskUpdate(task);
        setActiveTaskId(task.id);

        if (token) {
          try {
            await fetchProfile(token, true);
          } catch (profileError) {
            console.warn('Не удалось обновить профиль после генерации', profileError);
          }
        }

        if (!isTaskEffectivelyComplete(task)) {
          const existingTimeout = pollTimers.current.get(task.id);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(task.id);
          }

          const timeoutId = window.setTimeout(() => {
            void refreshTask(task.id, 1);
          }, POLL_INTERVAL_MS);

          pollTimers.current.set(task.id, timeoutId);
        }
      } catch (error) {
        console.error('Ошибка генерации cover', error);
        if (error && typeof error === 'object' && 'status' in error && (error as { status?: number }).status === 401) {
          localStorage.removeItem('authToken');
          setToken(null);
          setUser(null);
          setAuthStatus('unauthenticated');
          setAuthError('Сессия истекла. Войдите снова, чтобы продолжить.');
          setIsAuthDialogOpen(true);
        }
        
        let message = 'Не удалось запустить генерацию cover.';
        if (error instanceof Error) {
          message = error.message;
        }
        
        setGenerationError(message);
        throw error;
      } finally {
        setIsGeneratingMusic(false);
      }
    },
    [apiFetch, applyTaskUpdate, fetchProfile, isAuthenticated, refreshTask, token],
  );

  const handleAddInstrumental = useCallback(
    async (audioId: string, options: { prompt?: string; style?: string }) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация.');
      }

      setIsGeneratingMusic(true);
      setGenerationError(null);

      const payload = {
        audioId,
        prompt: options.prompt ?? null,
        style: options.style ?? null,
      };

      try {
        const response = await apiFetch('/music/add-instrumental', {
          method: 'POST',
          body: JSON.stringify(payload),
        });

        const task = normalizeGenerationResponse(response);
        
        if (!task.id || task.id === 'undefined') {
          throw new Error('Сервер не вернул идентификатор задачи. Попробуйте снова.');
        }
        
        applyTaskUpdate(task);
        setActiveTaskId(task.id);

        if (token) {
          try {
            await fetchProfile(token, true);
          } catch (profileError) {
            console.warn('Не удалось обновить профиль после генерации', profileError);
          }
        }

        if (!isTaskEffectivelyComplete(task)) {
          const existingTimeout = pollTimers.current.get(task.id);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(task.id);
          }

          const timeoutId = window.setTimeout(() => {
            void refreshTask(task.id, 1);
          }, POLL_INTERVAL_MS);

          pollTimers.current.set(task.id, timeoutId);
        }
      } catch (error) {
        console.error('Ошибка добавления инструментала', error);
        if (error && typeof error === 'object' && 'status' in error && (error as { status?: number }).status === 401) {
          localStorage.removeItem('authToken');
          setToken(null);
          setUser(null);
          setAuthStatus('unauthenticated');
          setAuthError('Сессия истекла. Войдите снова, чтобы продолжить.');
          setIsAuthDialogOpen(true);
        }
        
        let message = 'Не удалось добавить инструментал.';
        if (error instanceof Error) {
          message = error.message;
        }
        
        setGenerationError(message);
        throw error;
      } finally {
        setIsGeneratingMusic(false);
      }
    },
    [apiFetch, applyTaskUpdate, fetchProfile, isAuthenticated, refreshTask, token],
  );

  const handleAddVocals = useCallback(
    async (audioId: string, options: { prompt?: string; style?: string; lyrics?: string; title?: string; tags?: string }) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация.');
      }

      setIsGeneratingMusic(true);
      setGenerationError(null);

      const payload = {
        audioId,
        prompt: options.prompt ?? null,
        style: options.style ?? null,
        lyrics: options.lyrics ?? null,
        title: options.title ?? null,
        tags: options.tags ?? null,
      };

      try {
        const response = await apiFetch('/music/add-vocals', {
          method: 'POST',
          body: JSON.stringify(payload),
        });

        const task = normalizeGenerationResponse(response);
        
        if (!task.id || task.id === 'undefined') {
          throw new Error('Сервер не вернул идентификатор задачи. Попробуйте снова.');
        }
        
        applyTaskUpdate(task);
        setActiveTaskId(task.id);

        if (token) {
          try {
            await fetchProfile(token, true);
          } catch (profileError) {
            console.warn('Не удалось обновить профиль после генерации', profileError);
          }
        }

        if (!isTaskEffectivelyComplete(task)) {
          const existingTimeout = pollTimers.current.get(task.id);
          if (existingTimeout) {
            window.clearTimeout(existingTimeout);
            pollTimers.current.delete(task.id);
          }

          const timeoutId = window.setTimeout(() => {
            void refreshTask(task.id, 1);
          }, POLL_INTERVAL_MS);

          pollTimers.current.set(task.id, timeoutId);
        }
      } catch (error) {
        console.error('Ошибка добавления голоса', error);
        if (error && typeof error === 'object' && 'status' in error && (error as { status?: number }).status === 401) {
          localStorage.removeItem('authToken');
          setToken(null);
          setUser(null);
          setAuthStatus('unauthenticated');
          setAuthError('Сессия истекла. Войдите снова, чтобы продолжить.');
          setIsAuthDialogOpen(true);
        }
        
        let message = 'Не удалось добавить голос.';
        if (error instanceof Error) {
          message = error.message;
        }
        
        setGenerationError(message);
        throw error;
      } finally {
        setIsGeneratingMusic(false);
      }
    },
    [apiFetch, applyTaskUpdate, fetchProfile, isAuthenticated, refreshTask, token],
  );

  const applyAuthPayload = useCallback(
    async (payload: any, telegram = false) => {
      console.info('[auth] Applying payload', payload);
      
      if (!payload || typeof payload !== 'object') {
        throw new Error('Неверный формат данных авторизации.');
      }

      const nextToken =
        'token' in payload && typeof payload.token === 'string'
          ? payload.token
          : undefined;

      if (!nextToken || typeof nextToken !== 'string') {
        console.error('[auth] Invalid token in payload:', payload);
        throw new Error('Не удалось получить токен авторизации.');
      }

      if ('user' in payload && payload.user && typeof payload.user === 'object') {
        const userData = payload.user as AuthUser;
        console.info('[auth] Using user from payload', userData);
        
        // Убеждаемся, что все необходимые поля есть
        if (!userData.id) {
          console.error('[auth] User data missing id:', userData);
          throw new Error('Данные пользователя неполные.');
        }

        setUser(userData);
        setToken(nextToken);
        localStorage.setItem('authToken', nextToken);
        setIsTelegramContext(Boolean(userData?.is_telegram) || telegram);
        setAuthStatus('authenticated');
        setAuthError(null);
        setIsAuthDialogOpen(false);
        console.info('[auth] Auth payload applied successfully');
      } else {
        console.log('[auth] No user in payload, fetching profile...');
        await fetchProfile(nextToken, telegram);
      }
    },
    [fetchProfile],
  );

  const handleTelegramWidgetAuth = useCallback(
    async (loginPayload: Record<string, unknown>) => {
      if (!loginPayload || typeof loginPayload !== 'object') {
        setAuthError('Не удалось получить данные авторизации Telegram.');
        return;
      }

      const params = new URLSearchParams();

      const allowedTopLevelKeys: string[] = [
        'auth_date',
        'hash',
        'query_id',
        'state',
        'id',
        'first_name',
        'last_name',
        'username',
        'photo_url',
      ];

      allowedTopLevelKeys.forEach((key) => {
        const value = (loginPayload as Record<string, unknown>)[key];
        if (value === undefined || value === null) {
          return;
        }

        params.set(key, String(value));
      });

      const userPayloadRaw =
        'user' in loginPayload && typeof (loginPayload as Record<string, unknown>).user === 'object'
          ? (loginPayload as Record<string, unknown>).user
          : null;

      if (userPayloadRaw && typeof userPayloadRaw === 'object') {
        const allowedUserKeys = [
          'id',
          'first_name',
          'last_name',
          'username',
          'photo_url',
          'language_code',
        ];

        const sanitizedUser: Record<string, unknown> = {};

        allowedUserKeys.forEach((key) => {
          const value = (userPayloadRaw as Record<string, unknown>)[key];
          if (value !== undefined && value !== null) {
            sanitizedUser[key] = value;
          }
        });

        if (Object.keys(sanitizedUser).length > 0) {
          params.set('user', JSON.stringify(sanitizedUser));
        }

        if (!params.has('auth_date')) {
          const authDate = (userPayloadRaw as Record<string, unknown>).auth_date;
          if (typeof authDate === 'string' || typeof authDate === 'number') {
            params.set('auth_date', String(authDate));
          }
        }

        if (!params.has('hash')) {
          const userHash = (userPayloadRaw as Record<string, unknown>).hash;
          if (typeof userHash === 'string') {
            params.set('hash', userHash);
          }
        }
      }

      const hashValue = params.get('hash');

      if (!hashValue) {
        setAuthError('Ответ Telegram не содержит подпись. Попробуйте снова.');
        setAuthStatus('unauthenticated');
        return;
      }

      setAuthError(null);
      setAuthStatus('loading');

      try {
        const data = await apiFetch(
          '/auth/telegram',
          {
            method: 'POST',
            body: JSON.stringify({
              init_data: params.toString(),
              mode: 'widget',
            }),
          },
          null,
        );

      console.info('[auth] Telegram auth success response', data);

        await applyAuthPayload(data, true);
        setIsAuthDialogOpen(false);
      } catch (error) {
      console.error('[auth] Telegram auth error', error);
        setAuthError(
          error instanceof Error
            ? error.message
            : 'Не удалось выполнить вход через Telegram.',
        );
        setAuthStatus('unauthenticated');
      }
    },
    [apiFetch, applyAuthPayload],
  );

  const handleVKIDError = useCallback((message: string) => {
    // Тихо обрабатываем ошибки VK ID, не показываем пользователю
    console.warn('[VK ID] Error:', message);
    // Не устанавливаем ошибку авторизации для VK ID ошибок
    // setAuthError(normalized);
    // setAuthStatus((current) =>
    //   current === 'authenticated' ? current : 'unauthenticated',
    // );
  }, []);

  const handleVKIDAuth = useCallback(
    async (payload: VKIDAuthPayload) => {
      setAuthError(null);
      setAuthStatus('loading');

      try {
        const response = await apiFetch(
          '/auth/vkid',
          {
            method: 'POST',
            body: JSON.stringify({
              device_id: payload.device_id,
              token: {
                access_token: payload.token.access_token,
                refresh_token: payload.token.refresh_token,
                token_type: payload.token.token_type,
                expires_in: payload.token.expires_in,
                user_id: payload.token.user_id,
                scope: payload.token.scope ?? undefined,
              },
              state: payload.state ?? payload.token.state ?? null,
            }),
          },
          null,
        );

        await applyAuthPayload(response, false);
      } catch (error) {
        console.error('VK ID auth error', error);
        const message =
          error instanceof Error
            ? error.message
            : 'Не удалось выполнить вход через VK ID.';
        handleVKIDError(message);
      }
    },
    [apiFetch, applyAuthPayload, handleVKIDError],
  );

  const handleDemoLogin = useCallback(async () => {
    console.log('[auth] Starting demo login...');
    console.log('[auth] API base candidates:', apiBaseCandidates);
    setAuthError(null);
    setAuthStatus('loading');

    try {
      console.log('[auth] Making demo login request to /auth/demo...');
      const response = await apiFetch(
        '/auth/demo',
        {
          method: 'POST',
        },
        null,
      );

      console.log('[auth] Demo login response received:', response);
      
      if (!response || typeof response !== 'object') {
        throw new Error('Неверный формат ответа от сервера.');
      }

      if (!response.token || typeof response.token !== 'string') {
        console.error('[auth] No token in response:', response);
        throw new Error('Сервер не вернул токен авторизации.');
      }

      if (!response.user || typeof response.user !== 'object') {
        console.error('[auth] No user in response:', response);
        throw new Error('Сервер не вернул данные пользователя.');
      }

      console.log('[auth] Applying auth payload...');
      await applyAuthPayload(response, false);
      console.log('[auth] Demo login successful');
      setIsAuthDialogOpen(false);
    } catch (error) {
      console.error('[auth] Demo auth error:', error);
      console.error('[auth] Error details:', {
        message: error instanceof Error ? error.message : String(error),
        status: (error as any)?.status,
        stack: error instanceof Error ? error.stack : undefined,
      });
      
      let errorMessage = 'Не удалось войти в демо-режиме.';
      
      if (error instanceof Error) {
        errorMessage = error.message;
        
        // Если это 404, значит демо режим отключен на сервере или неправильный URL
        if (error.message.includes('404') || error.message.includes('Not Found') || (error as any)?.status === 404) {
          errorMessage = 'Демо-режим недоступен. Проверьте, что вы используете правильный адрес сайта.';
        } else if (error.message.includes('Failed to fetch') || error.message.includes('Network')) {
          errorMessage = 'Не удалось подключиться к серверу. Проверьте подключение к интернету.';
        }
      }
      
      setAuthError(errorMessage);
      setAuthStatus('unauthenticated');
    }
  }, [apiFetch, applyAuthPayload, apiBaseCandidates]);

  useEffect(() => {
    if (typeof window === 'undefined') {
      return;
    }

    const handleTelegramMessage = (event: MessageEvent) => {
      if (event.origin !== window.location.origin) {
        return;
      }

      const data = event.data;
      if (!data || typeof data !== 'object' || data.source !== 'telegram-auth-callback') {
        return;
      }

      if ('error' in data && data.error) {
        const message =
          typeof data.error === 'string'
            ? data.error
            : 'Не удалось завершить авторизацию через Telegram.';
        setAuthError(message);
        setAuthStatus('unauthenticated');
        return;
      }

      if ('payload' in data && data.payload) {
        void handleTelegramWidgetAuth(data.payload as Record<string, unknown>);
      }
    };

    window.addEventListener('message', handleTelegramMessage);

    return () => {
      window.removeEventListener('message', handleTelegramMessage);
    };
  }, [handleTelegramWidgetAuth]);

  const attemptTelegramLogin = useCallback(async () => {
    const telegram = window.Telegram?.WebApp;

    if (!telegram || !telegram.initData) {
      setIsTelegramContext(false);
      setAuthStatus('unauthenticated');
      setIsAuthDialogOpen(false);
      return;
    }

    setIsTelegramContext(true);
    setAuthError(null);
    setAuthStatus('loading');

    try {
      if (typeof telegram.ready === 'function') {
        telegram.ready();
      }

      telegram.expand?.();

      const data = await apiFetch(
        '/auth/telegram',
        {
          method: 'POST',
          body: JSON.stringify({ init_data: telegram.initData }),
        },
        null,
      );

      await applyAuthPayload(data, true);
    } catch (error) {
      setAuthError(
        error instanceof Error
          ? error.message
          : 'Авторизация через Telegram не удалась.',
      );
      setAuthStatus('unauthenticated');
      setIsAuthDialogOpen(false);
    }
  }, [apiFetch, applyAuthPayload]);

  const attemptTelegramLoginRef = useRef<(() => Promise<void>) | null>(null);
  const fetchProfileRef =
    useRef<((token: string, telegram?: boolean) => Promise<void>) | null>(null);

  useEffect(() => {
    attemptTelegramLoginRef.current = attemptTelegramLogin;
  }, [attemptTelegramLogin]);

  useEffect(() => {
    fetchProfileRef.current = fetchProfile;
  }, [fetchProfile]);

  useEffect(() => {
    let cancelled = false;

    const initialize = async () => {
      const storedToken = localStorage.getItem('authToken');

      if (storedToken && fetchProfileRef.current) {
        try {
          await fetchProfileRef.current(storedToken);
          return;
        } catch {
          localStorage.removeItem('authToken');
        }
      }

      if (!cancelled && attemptTelegramLoginRef.current) {
        await attemptTelegramLoginRef.current();
        setAuthStatus((current) =>
          current === 'authenticated' ? current : 'unauthenticated',
        );
      }
    };

    initialize();

    return () => {
      cancelled = true;
    };
  }, []);

  const handlePlaySong = (song: Song) => {
    setCurrentSong(song);
    const playableSource =
      song.streamUrl ??
      song.audioUrl ??
      song.sourceStreamUrl ??
      song.sourceAudioUrl;

    if (!playableSource) {
      setIsPlaying(false);
      setPlaybackError('Трек еще генерируется. Попробуйте снова через несколько секунд.');
      return;
    }

    setPlaybackError(null);

    if (currentSong?.id === song.id) {
      setIsPlaying((previous) => !previous);
    } else {
      setIsPlaying(true);
    }
  };

  const handleTogglePlay = () => {
    if (!currentSong) {
      return;
    }

    const playableSource =
      currentSong.streamUrl ??
      currentSong.audioUrl ??
      currentSong.sourceStreamUrl ??
      currentSong.sourceAudioUrl;

    if (!playableSource) {
      setPlaybackError('Трек еще в обработке. Мы начнем проигрывание, как только поток станет доступен.');
      setIsPlaying(false);
      return;
    }

    setPlaybackError(null);
    setIsPlaying((previous) => !previous);
  };

  const handleLogout = useCallback(async () => {
    setAuthStatus('loading');
    setAuthError(null);

    try {
      if (token) {
        await apiFetch('/auth/logout', { method: 'POST' });
      }
    } catch {
      // игнорируем сетевые ошибки при выходе
    }

    localStorage.removeItem('authToken');
    setToken(null);
    setUser(null);
    setIsAuthDialogOpen(false);

    if (isTelegramContext) {
      await attemptTelegramLogin();
    } else {
      setAuthStatus('unauthenticated');
    }
  }, [apiFetch, attemptTelegramLogin, isTelegramContext, token]);

  const retryTelegramLogin = useCallback(() => {
    setAuthError(null);

    if (typeof window !== 'undefined' && window.Telegram?.WebApp) {
      setAuthStatus('loading');
      setIsAuthDialogOpen(true);
      void attemptTelegramLogin();
      return;
    }

    const computedBotName =
      (loginBotName && loginBotName.trim().length > 0
        ? loginBotName.trim()
        : null) ??
      (telegramLoginBot && telegramLoginBot.trim().length > 0
        ? telegramLoginBot.trim()
        : null) ??
      FALLBACK_TELEGRAM_LOGIN_BOT;
    const computedBotId =
      typeof loginBotId === 'number' && !Number.isNaN(loginBotId)
        ? loginBotId
        : FALLBACK_TELEGRAM_BOT_ID;

    if (!loginBotName) {
      setLoginBotName(computedBotName);
    }

    if (!loginBotId) {
      setLoginBotId(computedBotId);
    }

    if (typeof window === 'undefined') {
      return;
    }

    const callbackUrl =
      typeof window !== 'undefined'
        ? `${window.location.origin}/backend/public/auth/telegram/callback`
        : null;

    const openLink = (url: string) => window.open(url, '_blank', 'noopener,noreferrer');

    setIsAuthDialogOpen(true);
    setAuthStatus('loading');

    if (computedBotId) {
      const origin = encodeURIComponent(window.location.origin);
      const redirect = callbackUrl ? `&redirect_to=${encodeURIComponent(callbackUrl)}` : '';
      const url = `https://oauth.telegram.org/auth?bot_id=${computedBotId}&origin=${origin}&embed=1&request_access=write${redirect}`;
      const tab = openLink(url);
      if (!tab) {
        const fallback = computedBotName
          ? `https://t.me/${computedBotName.replace(/^@/, '')}?start=webapp`
          : url;
        openLink(fallback);
      }
      return;
    }

    const normalized = computedBotName.replace(/^@/, '');
    const deepLink = `tg://resolve?domain=${normalized}&start=webapp`;
    const httpsLink =
      callbackUrl !== null && computedBotId
        ? `https://oauth.telegram.org/auth?bot_id=${encodeURIComponent(
            computedBotId,
          )}&origin=${encodeURIComponent(window.location.origin)}&request_access=write&redirect_to=${encodeURIComponent(
            callbackUrl,
          )}`
        : `https://t.me/${normalized}?start=webapp`;

    const newWindow = openLink(deepLink);
    if (!newWindow) {
      openLink(httpsLink);
    }
  }, [attemptTelegramLogin, loginBotId, loginBotName, telegramLoginBot]);

  const openAuthDialog = useCallback(() => {
      setAuthError(null);
      setIsAuthDialogOpen(true);
      if (authStatus !== 'authenticated') {
        setAuthStatus('unauthenticated');
      }
  }, [authStatus]);

  const closeAuthDialog = useCallback(() => {
    if (authStatus === 'loading') {
      return;
    }

    setIsAuthDialogOpen(false);
    setAuthError(null);
  }, [authStatus]);

  const effectiveLoginBot =
    (loginBotName ?? telegramLoginBot ?? FALLBACK_TELEGRAM_LOGIN_BOT)?.trim() ??
    FALLBACK_TELEGRAM_LOGIN_BOT;

  const toggleFavorite = useCallback((song: Song) => {
    setFavorites((previous) => {
      const exists = previous.some((item) => item.id === song.id);
      if (exists) {
        return previous.filter((item) => item.id !== song.id);
      }

      return [{ ...song }, ...previous];
    });
  }, []);

  const isSongFavorite = useCallback(
    (songId: string | undefined | null) =>
      Boolean(songId && favorites.some((item) => item.id === songId)),
    [favorites],
  );

  const loadPublicSongs = useCallback(
    async (filter: string = 'all') => {
      try {
        const filterMap: Record<string, string> = {
          'Выбор редакции': 'editorial',
          'Сегодня': 'today',
          'Неделя': 'week',
          'Месяц': 'month',
          'Все': 'all',
          'В эфире': 'live',
        };

        const apiFilter = filterMap[filter] || 'all';
        const response = await apiFetch(`/songs/public?filter=${apiFilter}&limit=100`, {}, null);
        
        if (response && response.songs && Array.isArray(response.songs)) {
          setPublicSongs(response.songs);
        }
      } catch (error) {
        console.error('Failed to load public songs:', error);
        setPublicSongs([]);
      }
    },
    [apiFetch],
  );

  const publishSong = useCallback(
    async (song: Song) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация для публикации трека.');
      }

      try {
        await apiFetch('/songs/publish', {
          method: 'POST',
          body: JSON.stringify({
            song_id: song.id,
            task_id: song.taskId,
          }),
        });

        // Update song in local state
        setSongs((prev) =>
          prev.map((s) =>
            s.id === song.id
              ? { ...s, isPublic: true, publishedAt: new Date().toISOString() }
              : s,
          ),
        );

        // Reload public songs if we're viewing them
        await loadPublicSongs('В эфире');

        return true;
      } catch (error) {
        console.error('Failed to publish song:', error);
        throw error;
      }
    },
    [apiFetch, isAuthenticated, loadPublicSongs],
  );

  const unpublishSong = useCallback(
    async (song: Song) => {
      if (!isAuthenticated) {
        throw new Error('Требуется авторизация для снятия трека с публикации.');
      }

      try {
        await apiFetch('/songs/unpublish', {
          method: 'POST',
          body: JSON.stringify({
            song_id: song.id,
          }),
        });

        // Update song in local state
        setSongs((prev) =>
          prev.map((s) =>
            s.id === song.id ? { ...s, isPublic: false, publishedAt: null } : s,
          ),
        );

        // Reload public songs
        await loadPublicSongs('В эфире');

        return true;
      } catch (error) {
        console.error('Failed to unpublish song:', error);
        throw error;
      }
    },
    [apiFetch, isAuthenticated, loadPublicSongs],
  );

  const sortedTasks = useMemo(() => {
    const list = Object.values(tasks) as GenerationTask[];
    return [...list].sort(
      (a, b) => Date.parse(b.updatedAt) - Date.parse(a.updatedAt),
    );
  }, [tasks]);

  const totalGenerations = useMemo(() => {
    if (typeof user?.generation_count === 'number') {
      return user.generation_count;
    }

    return sortedTasks.length;
  }, [sortedTasks, user]);

  const tokenBalance = useMemo(() => {
    if (typeof user?.token_balance === 'number') {
      return user.token_balance;
    }

    return 0;
  }, [user]);

  const latestTask =
    (activeTaskId && tasks[activeTaskId]) ?? sortedTasks[0] ?? null;

  const renderContent = () => {
    if (isDesktop && activeTab === 'home') {
      return (
        <DesktopHomePage
          songs={songs}
          publicSongs={publicSongs}
          onPlaySong={handlePlaySong}
          playingSongId={isPlaying ? currentSong?.id : null}
          onLoadPublicSongs={loadPublicSongs}
          onPublishSong={publishSong}
          onUnpublishSong={unpublishSong}
          isAuthenticated={isAuthenticated}
          onGenerate={handleGenerateMusic}
          isGenerating={isGeneratingMusic}
          latestTask={latestTask}
          generationError={generationError}
          onExtendSong={handleExtendSong}
          extendingSongId={extendingSongId}
          onRequireAuth={openAuthDialog}
        />
      );
    }

    switch (activeTab) {
      case 'home':
        return (
          <HomePage
            songs={songs}
            onPlaySong={handlePlaySong}
            playingSongId={isPlaying ? currentSong?.id : null}
          />
        );
      case 'create':
        return (
          <CreatePage
            isAuthenticated={isAuthenticated}
            onRequireAuth={openAuthDialog}
            onGenerate={handleGenerateMusic}
            onCover={handleCoverGeneration}
            onAddInstrumental={handleAddInstrumental}
            onAddVocals={handleAddVocals}
            isGenerating={isGeneratingMusic}
            latestTask={latestTask}
            generationError={generationError}
            onExtendSong={handleExtendSong}
            extendingSongId={extendingSongId}
            songs={songs}
          />
        );
      case 'library':
        return (
          <LibraryPage
            songs={songs}
            onPlaySong={handlePlaySong}
            currentSongId={currentSong?.id ?? null}
            isPlaying={isPlaying}
            onPublishSong={publishSong}
            onUnpublishSong={unpublishSong}
            isAuthenticated={isAuthenticated}
          />
        );
      case 'more':
        return (
          <MorePage
            isAuthenticated={isAuthenticated}
            user={user}
            onLogin={openAuthDialog}
            onLogout={user && !isTelegramContext ? () => void handleLogout() : undefined}
            isTelegramContext={isTelegramContext}
            generationCount={totalGenerations}
            tokenBalance={tokenBalance}
          />
        );
      default:
        return (
          <HomePage
            songs={songs}
            onPlaySong={handlePlaySong}
            playingSongId={isPlaying ? currentSong?.id : null}
          />
        );
    }
  };

  const showAuthLoader = authStatus === 'loading';

  // Desktop layout
  if (isDesktop) {
    return (
      <div className="flex h-screen flex-col bg-[#1a1b24] font-sans text-gray-300">
        <DesktopHeader
          user={user}
          onLogin={openAuthDialog}
          onLogout={user && !isTelegramContext ? () => void handleLogout() : undefined}
          onGenerate={() => setActiveTab('create')}
          isAuthenticating={authStatus === 'loading' && isAuthDialogOpen}
        />
        <div className="flex flex-1 overflow-hidden">
          <DesktopSidebar
            activeTab={activeTab}
            setActiveTab={setActiveTab}
            isAuthenticated={isAuthenticated}
            onAuthClick={openAuthDialog}
            onCreateClick={() => {
              // Переключаемся на вкладку playlists, если не на ней
              if (activeTab !== 'playlists') {
                setActiveTab('playlists');
              }
              // Открываем модальное окно
              setIsCreateModalOpen(true);
            }}
          />
          <main className="ml-64 flex-1 overflow-hidden pt-0">{renderContent()}</main>
        </div>
        {currentSong && (
          <DesktopPlayer
            song={currentSong}
            isPlaying={isPlaying}
            onTogglePlay={handleTogglePlay}
            isFavorite={isSongFavorite(currentSong.id)}
            onToggleFavorite={() => toggleFavorite(currentSong)}
          />
        )}
        <footer className="border-t border-white/10 bg-[#1a1b24] py-3 text-center">
          <p className="text-xs text-gray-500">ИП Абрамов В.В.</p>
        </footer>
        {playbackError && (
          <div className="fixed bottom-20 left-1/2 z-30 -translate-x-1/2 rounded-xl border border-red-500/40 bg-red-500/10 px-4 py-3 text-sm text-red-200 shadow-lg backdrop-blur">
            {playbackError}
          </div>
        )}
        {/* Create Modal - всегда доступна */}
        {isCreateModalOpen && (
          <div 
            className="fixed inset-0 z-50 flex items-center justify-center bg-black/70 px-4 backdrop-blur-sm"
            onClick={(e) => {
              if (e.target === e.currentTarget && !isGeneratingMusic) {
                setIsCreateModalOpen(false);
              }
            }}
          >
            <div 
              className="relative w-full max-w-2xl max-h-[90vh] overflow-y-auto rounded-2xl bg-[#1a1b24] shadow-2xl shadow-purple-500/20"
              onClick={(e) => e.stopPropagation()}
            >
              {/* Modal Header */}
              <div className="sticky top-0 z-10 flex items-center justify-between border-b border-white/10 bg-[#1a1b24] px-6 py-4">
                <h2 className="text-2xl font-bold text-white">Создать трек</h2>
                <button
                  onClick={() => setIsCreateModalOpen(false)}
                  disabled={isGeneratingMusic}
                  className="rounded-full p-2 text-gray-400 transition hover:bg-white/10 hover:text-white disabled:cursor-not-allowed disabled:opacity-50"
                  aria-label="Закрыть"
                >
                  ×
                </button>
              </div>

              {/* Modal Content */}
              <div className="p-6">
                <CreatePage
                  isAuthenticated={isAuthenticated}
                  onRequireAuth={() => {
                    openAuthDialog();
                    setIsCreateModalOpen(false);
                  }}
                  onGenerate={async (payload) => {
                    await handleGenerateMusic(payload);
                    // Не закрываем модалку автоматически, чтобы видеть прогресс
                  }}
                  isGenerating={isGeneratingMusic}
                  latestTask={latestTask}
                  generationError={generationError}
                  onExtendSong={handleExtendSong}
                  extendingSongId={extendingSongId}
                />
              </div>
            </div>
          </div>
        )}


        {isAuthDialogOpen && (
          <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/70 px-4">
            <div className="relative w-full max-w-sm space-y-6 rounded-2xl bg-[#171620] px-6 py-8 shadow-xl shadow-purple-500/20">
              <button
                type="button"
                onClick={closeAuthDialog}
                disabled={authStatus === 'loading'}
                className="absolute right-4 top-4 text-xl text-gray-500 transition hover:text-gray-200 disabled:cursor-not-allowed disabled:opacity-40"
                aria-label="Закрыть"
              >
                ×
              </button>
              <div className="mt-2 text-center space-y-1">
                <h2 className="text-2xl font-bold text-white">Melodist</h2>
                <p className="text-sm text-gray-400">
                  Войдите через Telegram, чтобы сохранять и генерировать музыку.
                </p>
              </div>
              {showAuthLoader ? (
                <div className="flex flex-col items-center gap-4 rounded-xl bg-black/40 px-6 py-8">
                  <div className="h-12 w-12 animate-spin rounded-full border-4 border-purple-500 border-t-transparent" />
                  <p className="text-center text-sm text-gray-400">
                    {isTelegramContext
                      ? 'Подтверждаем вход через Telegram...'
                      : 'Обрабатываем запрос...'}
                  </p>
                </div>
              ) : isTelegramContext && !authError ? (
                <div className="space-y-4 rounded-xl bg-black/40 px-6 py-8 text-center">
                  <p className="text-sm text-gray-300">
                    Ожидаем подтверждение запуска мини-приложения в Telegram.
                  </p>
                  <button
                    type="button"
                    onClick={retryTelegramLogin}
                    className="w-full rounded-full bg-purple-500 py-2 text-sm font-semibold text-white transition hover:bg-purple-400"
                  >
                    Попробовать снова
                  </button>
                </div>
              ) : (
                <BrowserTelegramLogin
                  botName={effectiveLoginBot}
                  botId={loginBotId ?? FALLBACK_TELEGRAM_BOT_ID}
                  onAuth={handleTelegramWidgetAuth}
                  onVKIDAuth={handleVKIDAuth}
                  onVKIDError={handleVKIDError}
                  isSubmitting={authStatus === 'loading'}
                  authError={authError}
                  onOpenTelegramApp={retryTelegramLogin}
                  vkidAppId={vkidAppId}
                  vkidRedirectUrl={vkidRedirectUrl}
                  showDemoLogin={demoLoginAvailable}
                  onDemoLogin={handleDemoLogin}
                />
              )}
            </div>
          </div>
        )}
      </div>
    );
  }

  // Mobile layout
  return (
    <div className="min-h-screen bg-[#1a1b24] font-sans text-gray-300">
      <Header
        user={user}
        onLogin={openAuthDialog}
        onLogout={user && !isTelegramContext ? () => void handleLogout() : undefined}
        isAuthenticating={authStatus === 'loading' && isAuthDialogOpen}
        showLogout={!isTelegramContext}
      />
      <main className="pb-32">{renderContent()}</main>
      {currentSong && (
        <MiniPlayer
          song={currentSong}
          isPlaying={isPlaying}
          onTogglePlay={handleTogglePlay}
          onOpenModal={() => setPlayerModalOpen(true)}
          isFavorite={isSongFavorite(currentSong.id)}
          onToggleFavorite={() => toggleFavorite(currentSong)}
        />
      )}
      {playbackError && (
        <div className="fixed bottom-32 left-4 right-4 z-30 rounded-xl border border-red-500/40 bg-red-500/10 px-4 py-3 text-sm text-red-200 shadow-lg backdrop-blur">
          {playbackError}
        </div>
      )}
      <BottomNav
        activeTab={activeTab}
        setActiveTab={setActiveTab}
        isAuthenticated={isAuthenticated}
        onAuthClick={openAuthDialog}
      />
      <footer className="border-t border-white/10 bg-[#1a1b24] py-4 text-center">
        <p className="text-xs text-gray-500">ИП Абрамов В.В.</p>
      </footer>
      {isPlayerModalOpen && currentSong && (
        <PlayerModal
          song={currentSong}
          isPlaying={isPlaying}
          onTogglePlay={handleTogglePlay}
          onClose={() => setPlayerModalOpen(false)}
        />
      )}
      {isAuthDialogOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/70 px-4">
          <div className="relative w-full max-w-sm space-y-6 rounded-2xl bg-[#171620] px-6 py-8 shadow-xl shadow-purple-500/20">
            <button
              type="button"
              onClick={closeAuthDialog}
              disabled={authStatus === 'loading'}
              className="absolute right-4 top-4 text-xl text-gray-500 transition hover:text-gray-200 disabled:cursor-not-allowed disabled:opacity-40"
              aria-label="Закрыть"
            >
              ×
            </button>
            <div className="mt-2 text-center space-y-1">
              <h2 className="text-2xl font-bold text-white">Melodist</h2>
              <p className="text-sm text-gray-400">
                Войдите через Telegram, чтобы сохранять и генерировать музыку.
              </p>
            </div>

            {showAuthLoader ? (
              <div className="flex flex-col items-center gap-4 rounded-xl bg-black/40 px-6 py-8">
                <div className="h-12 w-12 animate-spin rounded-full border-4 border-purple-500 border-t-transparent" />
                <p className="text-center text-sm text-gray-400">
                  {isTelegramContext
                    ? 'Подтверждаем вход через Telegram...'
                    : 'Обрабатываем запрос...'}
                </p>
              </div>
            ) : isTelegramContext && !authError ? (
              <div className="space-y-4 rounded-xl bg-black/40 px-6 py-8 text-center">
                <p className="text-sm text-gray-300">
                  Ожидаем подтверждение запуска мини-приложения в Telegram.
                </p>
                <button
                  type="button"
                  onClick={retryTelegramLogin}
                  className="w-full rounded-full bg-purple-500 py-2 text-sm font-semibold text-white transition hover:bg-purple-400"
                >
                  Попробовать снова
                </button>
              </div>
            ) : (
              <BrowserTelegramLogin
                botName={effectiveLoginBot}
                botId={loginBotId ?? FALLBACK_TELEGRAM_BOT_ID}
                onAuth={handleTelegramWidgetAuth}
                onVKIDAuth={handleVKIDAuth}
                onVKIDError={handleVKIDError}
                isSubmitting={authStatus === 'loading'}
                authError={authError}
                onOpenTelegramApp={retryTelegramLogin}
                vkidAppId={vkidAppId}
                vkidRedirectUrl={vkidRedirectUrl}
                showDemoLogin={demoLoginAvailable}
                onDemoLogin={handleDemoLogin}
              />
            )}
                </div>
        </div>
      )}
    </div>
  );
};

export default App;
