<?php

namespace App\Services;

use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;

class VKIDClient
{
    protected string $domain;
    protected int $appId;

    public function __construct()
    {
        $config = Config::get('services.vkid', []);

        $this->domain = (string) ($config['domain'] ?? 'id.vk.com');
        $this->appId = (int) ($config['app_id'] ?? 0);
    }

    /**
     * Fetch VK ID user info using access token.
     *
     * @return array<string, mixed>
     *
     * @throws \RuntimeException
     */
    public function fetchUserInfo(string $accessToken): array
    {
        if ($this->appId === 0) {
            throw new \RuntimeException('VK ID configuration is missing an app id.');
        }

        try {
            $response = Http::asForm()
                ->withQueryParameters([
                    'client_id' => $this->appId,
                ])
                ->post(sprintf('https://%s/oauth2/user_info', $this->domain), [
                    'access_token' => $accessToken,
                ])
                ->throw();
        } catch (RequestException|ConnectionException $exception) {
            $message = $exception->getMessage();

            throw new \RuntimeException("VK ID user info request failed: {$message}", previous: $exception);
        }

        $payload = $response->json();

        if (! is_array($payload)) {
            throw new \RuntimeException('VK ID user info response is malformed.');
        }

        $user = Arr::get($payload, 'user');

        if (is_array($user)) {
            return $user;
        }

        return [];
    }
}











