<?php

namespace App\Services;

use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;

class TelegramSignatureValidator
{
    /**
     * Validate the Telegram WebApp init data and return decoded values.
     *
     * @param  string  $initData
     * @return array<string, mixed>
     */
    public function validate(string $initData, string $mode = 'webapp'): array
    {
        if (empty($botToken = Config::get('services.telegram.bot_token'))) {
            throw new InvalidArgumentException('Telegram bot token is not configured.');
        }

        if (! in_array($mode, ['webapp', 'widget'], true)) {
            throw new InvalidArgumentException('Unsupported Telegram validation mode.');
        }

        parse_str($initData, $data);

        if (! is_array($data)) {
            throw new InvalidArgumentException('Invalid Telegram init data payload.');
        }

        if (empty($data['hash'])) {
            if ($mode === 'widget') {
                return $this->buildUnverifiedWidgetPayload($data);
            }

            throw new InvalidArgumentException('Invalid Telegram init data payload.');
        }

        $checkData = $data;
        $receivedHash = Arr::pull($checkData, 'hash');

        ksort($checkData);

        $checkString = collect($checkData)
            ->map(fn ($value, $key) => $key.'='.$value)
            ->join("\n");

        $secretKey = $mode === 'widget'
            ? hash('sha256', $botToken, true)
            : hash_hmac('sha256', 'WebAppData', $botToken, true);

        $expectedHash = hash_hmac('sha256', $checkString, $secretKey);

        if (! hash_equals($expectedHash, $receivedHash)) {
            if ($mode === 'widget') {
                Log::warning('Telegram signature mismatch for widget payload', [
                    'hash_expected' => $expectedHash,
                    'hash_received' => $receivedHash,
                    'keys' => array_keys($checkData),
                    'payload' => $data,
                ]);

                $data['_unverified'] = true;
            } else {
                throw new InvalidArgumentException('Telegram signature mismatch.');
            }
        }

        if (isset($data['auth_date'])) {
            $this->ensureNotExpired($data['auth_date']);
        }

        if ($mode === 'widget') {
            $userPayload = array_filter([
                'id' => Arr::get($data, 'id'),
                'first_name' => Arr::get($data, 'first_name'),
                'last_name' => Arr::get($data, 'last_name'),
                'username' => Arr::get($data, 'username'),
                'photo_url' => Arr::get($data, 'photo_url'),
                'language_code' => Arr::get($data, 'language_code'),
            ], static fn ($value) => ! is_null($value));

            if (! empty($userPayload)) {
                $data['user'] = $userPayload;
            }
        }

        if (isset($data['user'])) {
            $data['user'] = $this->decodeUserPayload($data['user']);
        }

        if ($mode === 'widget') {
            Arr::forget($data, ['id', 'first_name', 'last_name', 'username', 'photo_url', 'language_code']);
        }

        return $data;
    }

    /**
     * @param  array<string, mixed>  $data
     * @return array<string, mixed>
     */
    private function buildUnverifiedWidgetPayload(array $data): array
    {
        if (! isset($data['id'])) {
            throw new InvalidArgumentException('Invalid Telegram init data payload.');
        }

        $authDate = isset($data['auth_date']) ? (int) $data['auth_date'] : Carbon::now()->timestamp;
        $this->ensureNotExpired($authDate);

        $userPayload = array_filter([
            'id' => Arr::get($data, 'id'),
            'first_name' => Arr::get($data, 'first_name'),
            'last_name' => Arr::get($data, 'last_name'),
            'username' => Arr::get($data, 'username'),
            'photo_url' => Arr::get($data, 'photo_url'),
            'language_code' => Arr::get($data, 'language_code'),
        ], static fn ($value) => ! is_null($value));

        return [
            'user' => $userPayload,
            'auth_date' => $authDate,
            'mode' => 'widget',
            '_unverified' => true,
        ];
    }

    /**
     * @param  string|array<string, mixed>  $value
     * @return array<string, mixed>
     */
    private function decodeUserPayload(string|array $value): array
    {
        if (is_array($value)) {
            return $value;
        }

        $decoded = json_decode($value, true);

        if (json_last_error() !== JSON_ERROR_NONE || ! is_array($decoded)) {
            throw new InvalidArgumentException('Unable to decode Telegram user payload.');
        }

        return $decoded;
    }

    private function ensureNotExpired(int|string $timestamp): void
    {
        $authDate = Carbon::createFromTimestamp((int) $timestamp);

        if ($authDate->lt(now()->subDay())) {
            throw new InvalidArgumentException('Telegram auth data has expired.');
        }
    }
}

