<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Song extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'user_id',
        'suno_song_id',
        'task_id',
        'title',
        'artist',
        'prompt',
        'tags',
        'artwork_url',
        'source_artwork_url',
        'audio_url',
        'source_audio_url',
        'stream_url',
        'source_stream_url',
        'source_audio_id',
        'status',
        'model',
        'duration_seconds',
        'is_public',
        'published_at',
        'plays_count',
        'likes_count',
    ];

    protected $casts = [
        'is_public' => 'boolean',
        'published_at' => 'datetime',
        'duration_seconds' => 'integer',
        'plays_count' => 'integer',
        'likes_count' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Get the user that owns the song.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include public songs.
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * Scope a query to filter by time period.
     */
    public function scopeInPeriod($query, string $period)
    {
        return match ($period) {
            'today' => $query->whereDate('published_at', today()),
            'week' => $query->where('published_at', '>=', now()->subWeek()),
            'month' => $query->where('published_at', '>=', now()->subMonth()),
            'all' => $query,
            default => $query,
        };
    }
}
