import React from 'react';
import { Song } from '../types';
import { VerifiedIcon, MoreHorizontalIcon, SoundWaveIcon } from './Icons';

interface SongItemProps {
  song: Song;
  isPlaying: boolean;
  onTogglePlay: (id: number) => void;
  rank?: string;
}

const SongItem: React.FC<SongItemProps> = ({ song, isPlaying, onTogglePlay, rank }) => {
  return (
    <div className="flex items-center space-x-4 p-2 rounded-lg hover:bg-[#30313d]/60 transition-colors">
      {rank && <div className="text-gray-400 font-bold text-lg w-6 text-center">{rank}</div>}
      <div className="relative flex-shrink-0">
          <img src={song.imageUrl} alt={song.title} className="w-16 h-16 rounded-lg object-cover" />
          <button 
            onClick={() => onTogglePlay(song.id)}
            className="absolute inset-0 bg-black/30 flex items-center justify-center rounded-lg opacity-0 hover:opacity-100 transition-opacity"
            aria-label={`Play ${song.title}`}
          >
            {isPlaying && <SoundWaveIcon className="text-white" />}
          </button>
          {isPlaying && (
              <div className="absolute inset-0 bg-black/50 flex items-center justify-center rounded-lg">
                  <SoundWaveIcon className="text-white" />
              </div>
          )}
      </div>
      <div className="flex-1 min-w-0" onClick={() => onTogglePlay(song.id)} role="button" tabIndex={0}>
        <h3 className="text-white font-semibold truncate">{song.title}</h3>
        <div className="flex items-center gap-1.5 mt-1">
            <img src={song.artistAvatar} alt={song.artist} className="w-4 h-4 rounded-full" />
            <p className="text-gray-400 text-sm truncate">{song.artist} <span className="text-gray-500 mx-1">•</span> {song.tags}</p>
            {song.artistVerified && <VerifiedIcon className="w-4 h-4 text-blue-500 flex-shrink-0" />}
        </div>
      </div>
      <button className="text-gray-400 hover:text-white">
        <MoreHorizontalIcon className="w-6 h-6" />
      </button>
    </div>
  );
};

export default SongItem;