import React, { useState } from 'react';
import { PlayIcon, PauseIcon, SpinnerIcon } from './Icons';

const genres = ["Электроника", "Хип-хоп", "Поп", "Рок", "Джаз", "Классика"];
const moods = ["Веселое", "Спокойное", "Энергичное", "Драматичное", "Романтичное"];

const Chip: React.FC<{ label: string; isSelected: boolean; onClick: () => void, disabled: boolean }> = ({ label, isSelected, onClick, disabled }) => (
  <button
    onClick={onClick}
    disabled={disabled}
    className={`px-4 py-2 text-sm font-semibold rounded-full transition-colors whitespace-nowrap ${
      isSelected ? 'bg-purple-600 text-white' : 'bg-[#30313d] text-gray-300'
    } ${disabled ? 'opacity-50 cursor-not-allowed' : 'hover:bg-purple-500'}`}
  >
    {label}
  </button>
);

const GeneratedTrack: React.FC<{ onReset: () => void }> = ({ onReset }) => {
    const [isPlaying, setIsPlaying] = useState(false);
    return (
        <div className="p-4 flex flex-col items-center justify-center text-center h-full animate-fade-in">
            <h2 className="text-3xl font-bold text-white mb-8">Ваш трек готов!</h2>
            <div className="relative w-64 h-64">
                <img src="https://picsum.photos/seed/generated/256/256" alt="Generated album art" className="rounded-2xl shadow-lg w-full h-full object-cover" />
                <div className="absolute inset-0 bg-black/30 rounded-2xl"></div>
            </div>
            <h3 className="text-xl font-bold mt-6">Сгенерированный трек 1</h3>
            <p className="text-gray-400">SLAVA AI</p>

             <button 
                onClick={() => setIsPlaying(!isPlaying)}
                className="mt-8 text-white bg-purple-600 rounded-full p-4 hover:bg-purple-700 transition-colors"
                aria-label={isPlaying ? 'Pause' : 'Play'}
            >
                {isPlaying ? <PauseIcon className="w-8 h-8"/> : <PlayIcon className="w-8 h-8"/>}
            </button>
            
            <button 
                onClick={onReset}
                className="mt-12 bg-gray-700 text-white font-bold py-3 px-6 rounded-full hover:bg-gray-600 transition-all"
            >
                Создать еще
            </button>
        </div>
    );
};


const CreatePage: React.FC = () => {
  const [selectedGenre, setSelectedGenre] = useState<string | null>(genres[0]);
  const [selectedMood, setSelectedMood] = useState<string | null>(moods[0]);
  const [duration, setDuration] = useState(60);
  const [prompt, setPrompt] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [generated, setGenerated] = useState(false);

  const handleGenerate = () => {
    setIsLoading(true);
    // Simulate API call
    setTimeout(() => {
      setIsLoading(false);
      setGenerated(true);
    }, 2000);
  };

  const handleReset = () => {
      setGenerated(false);
      // Optional: reset form fields
      // setSelectedGenre(genres[0]);
      // setSelectedMood(moods[0]);
      // setDuration(60);
      // setPrompt("");
  }

  if (generated) {
      return <GeneratedTrack onReset={handleReset} />;
  }

  return (
    <div className="p-4">
      <h2 className="text-3xl font-bold text-white mb-6">Создать музыку</h2>
      
      <fieldset disabled={isLoading} className="space-y-6">
        <div>
          <label className="block text-lg font-semibold text-gray-300 mb-3">Жанр</label>
          <div className="flex flex-wrap gap-2">
            {genres.map(genre => <Chip key={genre} label={genre} isSelected={selectedGenre === genre} onClick={() => setSelectedGenre(genre)} disabled={isLoading} />)}
          </div>
        </div>

        <div>
          <label className="block text-lg font-semibold text-gray-300 mb-3">Настроение</label>
          <div className="flex flex-wrap gap-2">
            {moods.map(mood => <Chip key={mood} label={mood} isSelected={selectedMood === mood} onClick={() => setSelectedMood(mood)} disabled={isLoading} />)}
          </div>
        </div>

        <div>
            <label htmlFor="duration-slider" className="flex justify-between items-center text-lg font-semibold text-gray-300 mb-3">
                <span>Длительность</span>
                <span className="text-purple-400 font-mono">{duration} сек</span>
            </label>
            <input 
                id="duration-slider"
                type="range" 
                min="30" 
                max="180" 
                step="5"
                value={duration}
                onChange={e => setDuration(parseInt(e.target.value))}
                className="w-full h-2 bg-gray-700 rounded-lg appearance-none cursor-pointer accent-purple-500"
                disabled={isLoading}
            />
        </div>

        <div>
            <label htmlFor="prompt-input" className="block text-lg font-semibold text-gray-300 mb-3">Промпт (необязательно)</label>
            <textarea 
                id="prompt-input"
                value={prompt}
                onChange={e => setPrompt(e.target.value)}
                placeholder="Например, синтвейв-трек для ночной поездки по киберпанк-городу"
                rows={3}
                className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
                disabled={isLoading}
            />
        </div>
      </fieldset>
      
      <div className="mt-8">
        <button 
          onClick={handleGenerate}
          disabled={isLoading}
          className="w-full bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-bold py-4 px-4 rounded-full flex items-center justify-center text-lg hover:opacity-90 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {isLoading ? (
            <>
              <SpinnerIcon className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" />
              Генерация...
            </>
          ) : (
            'Сгенерировать'
          )}
        </button>
      </div>
    </div>
  );
};

export default CreatePage;